# EduMetrix UK Platform - Project Structure

## 📁 Root Directory Structure

```
EduMetrix UK Platform/
├── 📄 server.js                 # Main Express server file
├── 📄 package.json              # Backend dependencies & scripts
├── 📄 package-lock.json         # Lock file for backend
├── 📄 .env                      # Environment variables
├── 📄 README.md                 # Project documentation
├── 📄 setup.sh                  # Automated setup script
├── 📄 test-connection.js        # Database connection test
├── 📂 client/                   # React frontend application
├── 📂 models/                   # MongoDB/Mongoose models
├── 📂 routes/                   # Express API routes
├── 📂 middleware/               # Custom middleware
├── 📂 utils/                    # Utility functions
├── 📂 uploads/                  # File uploads directory
└── 📂 node_modules/             # Backend dependencies
```

## 📂 Backend Structure

### 🗄️ Models (`/models/`) - Organized by User Type
```
models/
├── 📂 admin/                    # Admin-specific models
│   ├── 📄 Invoice.js            # Finance/billing model
│   └── 📄 TrialRequest.js       # Trial class requests model
├── 📂 student/                  # Student-specific models
│   └── 📄 Student.js            # Student profile model
├── 📂 teacher/                  # Teacher-specific models
│   └── 📄 Teacher.js            # Teacher profile model
└── 📂 shared/                   # Shared models across user types
    ├── 📄 User.js               # Base user authentication model
    ├── 📄 Class.js              # Class scheduling model
    └── 📄 Chat.js               # Real-time chat model
```

### 🛣️ Routes (`/routes/`) - Organized by User Type
```
routes/
├── 📂 admin/                    # Admin-specific routes
│   ├── 📄 admin.js              # Admin management endpoints
│   └── 📄 finance.js            # Finance/invoice endpoints
├── 📂 student/                  # Student-specific routes
│   └── 📄 student.js            # Student-specific endpoints
├── 📂 teacher/                  # Teacher-specific routes
│   └── 📄 teacher.js            # Teacher-specific endpoints
└── 📂 shared/                   # Shared routes across user types
    ├── 📄 auth.js               # Authentication endpoints
    ├── 📄 classes.js            # Class management endpoints
    └── 📄 chat.js               # Real-time chat endpoints
```

### 🔧 Middleware (`/middleware/`)
```
middleware/
└── 📄 auth.js                   # JWT authentication middleware
```

### 🛠️ Utils (`/utils/`)
```
utils/
├── 📄 passwordGenerator.js      # Auto-generate user credentials
└── 📄 syllabusHelper.js         # Country-specific syllabus mapping
```

## 📂 Frontend Structure (`/client/`)

### ⚛️ React App Structure
```
client/
├── 📄 package.json              # Frontend dependencies & scripts
├── 📄 package-lock.json         # Lock file for frontend
├── 📂 public/                   # Static assets
├── 📂 src/                      # React source code
└── 📂 node_modules/             # Frontend dependencies
```

### 📱 Source Code (`/client/src/`)
```
src/
├── 📄 App.js                    # Main React application
├── 📄 index.js                  # React DOM entry point
├── 📂 components/               # Reusable React components
├── 📂 pages/                    # Page components
├── 📂 context/                  # React Context providers
├── 📂 services/                 # API service functions
└── 📂 utils/                    # Frontend utilities
```

### 🧩 Components (`/client/src/components/`)
```
components/
├── 📄 Layout.js                 # Main layout with navigation
├── 📄 ProtectedRoute.js         # Route protection component
└── 📄 ChatSystem.js             # Real-time chat component
```

### 📄 Pages (`/client/src/pages/`) - Organized by User Type
```
pages/
├── 📄 Login.js                  # Login page
├── 📄 Dashboard.js              # Main dashboard
├── 📂 admin/                    # Admin-specific pages
├── 📂 student/                  # Student-specific pages
└── 📂 teacher/                  # Teacher-specific pages
    └── 📄 TeacherDashboard.js   # Teacher dashboard with statistics
```

### 👨‍💼 Admin Pages (`/client/src/pages/admin/`)
```
admin/
├── 📄 StudentCreate.js          # Create student profiles
├── 📄 StudentsList.js           # List/manage students
├── 📄 TeacherCreate.js          # Create teacher profiles
├── 📄 TeachersList.js           # List/manage teachers
├── 📄 ClassScheduler.js         # Schedule classes with recurring options
├── 📄 CalendarView.js           # Calendar view (day/week/month)
└── 📄 FinanceDashboard.js       # Invoice & finance management
```

### 👨‍🎓 Student Pages (`/client/src/pages/student/`)
```
student/
└── 📄 StudentDashboard.js       # Student dashboard with class info
```

### 👨‍🏫 Teacher Pages (`/client/src/pages/teacher/`)
```
teacher/
└── 📄 TeacherDashboard.js       # Teacher dashboard with daily/monthly statistics
```

### 🔌 Services (`/client/src/services/`)
```
services/
└── 📄 api.js                    # API endpoints & Axios configuration
```

### 🏪 Context (`/client/src/context/`)
```
context/
└── 📄 AuthContext.js            # Authentication state management
```

## 🗃️ Database Models

### 👤 User Model
- Base authentication for all user types
- Auto-generated credentials system
- Role-based access control

### 👨‍🎓 Student Model
- Personal & parent information
- Country-specific syllabus options
- Subject selections
- Google Meet links
- Hourly rates & currency

### 👨‍🏫 Teacher Model
- Personal & contact information
- Subject expertise
- Banking details for payments
- Availability schedule
- Salary structure (fixed/hourly)

### 📅 Class Model
- Class scheduling with timezone support
- Recurring class options (daily/weekly/custom)
- Attendance tracking
- Meeting links
- Rescheduling capabilities

### 💬 Chat Model
- Real-time messaging between users
- Subject-specific chat restrictions
- 31-day message retention
- Admin monitoring capabilities

### 🧾 Invoice Model
- Automatic invoice generation
- Monthly class summaries
- Multi-currency support
- Payment tracking

### 📝 Trial Request Model
- Trial class bookings
- Manual & website submissions
- Status tracking

## 🚀 Key Features

### 🔐 Authentication System
- Auto-generated usernames: `name@edumetrix.uk`
- Secure password generation with symbols
- JWT-based authentication
- Role-based access control

### 👨‍💼 Admin Features
- Complete student/teacher management
- Class scheduling with recurring options
- Calendar management (day/week/month views)
- Finance tracking & invoice generation
- Real-time chat monitoring
- Soft delete & restore functionality

### 👨‍🎓 Student Features
- Dashboard with class schedules
- Google Meet class joining
- Live chat with assigned teachers
- Homework & assignment viewing
- Invoice & payment tracking

### 👨‍🏫 Teacher Features
- Dashboard with daily/monthly statistics
- Class management & attendance
- Student assignments by subject
- Live chat with students & admin
- Monthly earnings reports

### 💰 Finance System
- Automatic invoice generation
- Monthly student reports
- Teacher earnings tracking
- Multi-currency support (USD, GBP, AED, INR)
- Payment status management

### 💬 Real-time Chat
- Socket.io implementation
- Subject-specific restrictions
- 31-day message retention
- Admin monitoring capabilities

### 🌍 International Support
- Multi-timezone class scheduling
- Country-specific syllabus mapping
- Multi-currency billing
- International phone number support

## 📦 Dependencies

### Backend Dependencies
- **express**: Web framework
- **mongoose**: MongoDB ODM
- **socket.io**: Real-time communication
- **jsonwebtoken**: JWT authentication
- **bcryptjs**: Password hashing
- **multer**: File upload handling
- **moment-timezone**: Timezone handling
- **cors**: Cross-origin resource sharing

### Frontend Dependencies
- **react**: Frontend framework
- **@mui/material**: UI component library
- **axios**: HTTP client
- **socket.io-client**: Real-time client
- **react-router-dom**: Client-side routing
- **moment**: Date/time handling

## 🏃‍♂️ Running the Application

1. **Backend**: `npm run server` (runs on port 5001)
2. **Frontend**: `npm run client` (runs on port 3000)
3. **Both**: `npm run dev` (runs both concurrently)

## 🔧 Configuration Files

- **`.env`**: Environment variables
- **`package.json`**: Dependencies & scripts
- **`setup.sh`**: Automated setup script
- **`test-connection.js`**: Database connection test

The platform is fully functional with all features implemented and ready for production use! 🎉