import mongoose from 'mongoose';
import moment from 'moment-timezone';

const classSchema = new mongoose.Schema({
  studentId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Student',
    required: true
  },
  teacherId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Teacher',
    required: true
  },
  subject: {
    type: String,
    enum: ['Mathematics', 'General Science', 'Physics', 'Chemistry', 'Biology', 'English', 'Hindi', 'Computer Science', 'Arabic', 'Social Science', 'Geography', 'Malayalam'],
    required: true
  },
  scheduledDate: {
    type: Date,
    required: true
  },
  startTime: {
    type: String,
    required: true
  },
  endTime: {
    type: String,
    required: true
  },
  duration: {
    type: Number, // Duration in minutes
    required: true,
    default: 60
  },
  timezone: {
    type: String,
    required: true,
    default: 'Asia/Dubai'
  },
  googleMeetLink: {
    type: String,
    required: true
  },
  classColor: {
    type: String,
    default: '#1976d2' // Material-UI primary blue
  },
  status: {
    type: String,
    enum: ['scheduled', 'ongoing', 'completed', 'cancelled', 'missed', 'rescheduled'],
    default: 'scheduled'
  },
  recurringSchedule: {
    isRecurring: {
      type: Boolean,
      default: false
    },
    frequency: {
      type: String,
      enum: ['daily', 'weekly', 'custom'],
      default: 'weekly'
    },
    weeklyDays: [{
      type: String,
      enum: ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday']
    }],
    endDate: {
      type: Date
    },
    parentClassId: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'Class'
    }
  },
  attendance: {
    studentJoined: {
      type: Boolean,
      default: false
    },
    teacherJoined: {
      type: Boolean,
      default: false
    },
    studentJoinTime: {
      type: Date
    },
    teacherJoinTime: {
      type: Date
    },
    studentLeaveTime: {
      type: Date
    },
    teacherLeaveTime: {
      type: Date
    },
    present: {
      type: Boolean,
      default: false
    },
    late: {
      type: Boolean,
      default: false
    },
    absent: {
      type: Boolean,
      default: false
    },
    notes: {
      type: String,
      default: ''
    },
    markedAt: {
      type: Date
    },
    markedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'Teacher'
    }
  },
  attendanceStatus: {
    studentAttended: {
      type: Boolean,
      default: null
    },
    teacherAttended: {
      type: Boolean,
      default: null
    }
  },
  classNotes: {
    type: String,
    default: null
  },
  homeworkAssigned: {
    type: String,
    default: null
  },
  notes: {
    type: String,
    default: ''
  },
  recording: {
    isRecorded: {
      type: Boolean,
      default: false
    },
    recordingUrl: {
      type: String
    }
  },
  homework: {
    assigned: {
      type: Boolean,
      default: false
    },
    description: {
      type: String
    },
    dueDate: {
      type: Date
    },
    completed: {
      type: Boolean,
      default: false
    }
  },
  // Reschedule tracking
  rescheduledFrom: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Class',
    default: null
  },
  rescheduledTo: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Class',
    default: null
  },
  rescheduleReason: {
    type: String,
    default: null
  },
  isActive: {
    type: Boolean,
    default: true
  },
  isDeleted: {
    type: Boolean,
    default: false
  },
  deletedAt: {
    type: Date,
    default: null
  },
  createdBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: false,
    default: null
  }
}, {
  timestamps: true
});

// Index for efficient queries
classSchema.index({ studentId: 1, scheduledDate: 1 });
classSchema.index({ teacherId: 1, scheduledDate: 1 });
classSchema.index({ scheduledDate: 1, status: 1 });

// Virtual for displaying class time in different timezones
classSchema.virtual('displayTime').get(function() {
  return moment(this.scheduledDate).tz(this.timezone);
});

// Method to get student's timezone display
classSchema.methods.getStudentDisplayTime = function(studentTimezone) {
  return moment(this.scheduledDate).tz(studentTimezone || this.timezone);
};

// Method to get teacher's timezone display (IST)
classSchema.methods.getTeacherDisplayTime = function() {
  return moment(this.scheduledDate).tz('Asia/Kolkata');
};

// Method to get admin's timezone display (IST)
classSchema.methods.getAdminDisplayTime = function() {
  return moment(this.scheduledDate).tz('Asia/Kolkata');
};

// Static method to create recurring classes
classSchema.statics.createRecurringClasses = async function(classData, options = {}) {
  const classes = [];

  if (!classData.recurringSchedule.isRecurring) {
    // Create single class
    const singleClass = await this.create(classData);
    return [singleClass];
  }

  // Parse the initial scheduled date in the class's timezone
  let currentDate = moment.tz(classData.scheduledDate, classData.timezone);
  const endDate = moment.tz(classData.recurringSchedule.endDate, classData.timezone);

  // Extract the time components from the original scheduled date
  const classHour = currentDate.hour();
  const classMinute = currentDate.minute();

  let count = 0;
  const maxClasses = options.maxClasses || 365; // Limit to prevent infinite loops

  while (currentDate.isSameOrBefore(endDate) && count < maxClasses) {
    let shouldCreateClass = false;

    if (classData.recurringSchedule.frequency === 'daily') {
      shouldCreateClass = true;
    } else if (classData.recurringSchedule.frequency === 'weekly') {
      // Get the day name in the class's timezone
      const dayName = currentDate.format('dddd');
      shouldCreateClass = classData.recurringSchedule.weeklyDays.includes(dayName);
    }

    if (shouldCreateClass) {
      // Create a moment in the class's timezone with the specific date and time
      const classDateTime = moment.tz(
        {
          year: currentDate.year(),
          month: currentDate.month(),
          date: currentDate.date(),
          hour: classHour,
          minute: classMinute
        },
        classData.timezone
      );

      const classInstance = {
        ...classData,
        // Convert to UTC for storage while preserving the exact local time
        scheduledDate: classDateTime.toDate(),
        recurringSchedule: {
          ...classData.recurringSchedule,
          parentClassId: classData._id || null
        }
      };

      try {
        const createdClass = await this.create(classInstance);
        classes.push(createdClass);
        count++;
      } catch (error) {
        console.error('Error creating recurring class:', error);
      }
    }

    currentDate.add(1, 'day');
  }

  return classes;
};

export default mongoose.models.Class || mongoose.model('Class', classSchema);