import mongoose from 'mongoose';

const studentSettingsSchema = new mongoose.Schema({
  studentId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Student',
    required: true,
    unique: true
  },

  // Appearance Settings
  appearance: {
    colorTheme: {
      type: String,
      enum: ['blue', 'purple', 'green', 'orange', 'pink', 'teal', 'cyan', 'indigo'],
      default: 'blue'
    },
    darkMode: {
      type: Boolean,
      default: false
    }
  },

  // Notification Settings
  notifications: {
    assignments: {
      type: Boolean,
      default: true
    },
    grades: {
      type: Boolean,
      default: true
    },
    exams: {
      type: Boolean,
      default: true
    },
    announcements: {
      type: Boolean,
      default: true
    },
    messages: {
      type: Boolean,
      default: true
    }
  },

  // Profile Settings
  profile: {
    name: {
      type: String,
      default: ''
    },
    email: {
      type: String,
      default: ''
    },
    phone: {
      type: String,
      default: ''
    },
    timezone: {
      type: String,
      default: 'UTC'
    }
  },

  // Privacy Settings
  privacy: {
    dataAnalytics: {
      type: Boolean,
      default: true
    },
    promotionalEmails: {
      type: Boolean,
      default: true
    }
  },

  // Language Settings
  language: {
    type: String,
    enum: ['en', 'es', 'fr', 'de', 'it', 'pt', 'zh', 'ja'],
    default: 'en'
  }
}, {
  timestamps: true
});

// Index for faster queries
studentSettingsSchema.index({ studentId: 1 });

// Static method to get default settings for a student
studentSettingsSchema.statics.getDefaultSettings = function(studentId) {
  return {
    studentId,
    appearance: {
      colorTheme: 'blue',
      darkMode: false
    },
    notifications: {
      assignments: true,
      grades: true,
      exams: true,
      announcements: true,
      messages: true
    },
    profile: {
      name: '',
      email: '',
      phone: '',
      timezone: 'UTC'
    },
    privacy: {
      dataAnalytics: true,
      promotionalEmails: true
    },
    language: 'en'
  };
};

// Method to update specific setting category
studentSettingsSchema.methods.updateCategory = function(category, updates) {
  if (this[category]) {
    Object.assign(this[category], updates);
  }
  return this.save();
};

export default mongoose.model('StudentSettings', studentSettingsSchema);