import express from 'express';
import mongoose from 'mongoose';
import multer from 'multer';
import path from 'path';
import Message from '../../models/shared/Message.js';
import Employee from '../../models/teacher/Teacher.js';
import Student from '../../models/student/Student.js';
import Conversation from '../../models/shared/Conversation.js';
import Admin from '../../models/admin/User.js';

const router = express.Router();

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, 'uploads/chat/');
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({
  storage,
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB limit
  fileFilter: (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif|pdf|doc|docx|xls|xlsx|txt|mp4|mp3|wav/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Invalid file type. Only images, documents, videos, and audio files are allowed.'));
    }
  }
});

const isAdmin = (req, res, next) => {
  req.user = { role: 'admin' };
  if (req.user && req.user.role === 'admin') {
    return next();
  }
  res.status(403).json({ message: 'Unauthorized' });
};

// Function to initialize message routes with io
export default function messageRouter(io, users){
  // Get all teachers (employees)

  router.get('/employee', isAdmin, async (req, res) => {
    const { adminId } = req.query;

    if (!adminId) {
      return res.status(400).json({ error: 'adminId is required' });
    }

    const adminObjectId = new mongoose.Types.ObjectId(adminId);

    try {
      // Select 'name' field from Teacher model (not 'employeeName')
      const teachers = await Employee.find({ isActive: true, isDeleted: false })
        .select('_id name email')
        .lean();

      console.log('Found teachers:', teachers.length);

      const enrichedTeachers = await Promise.all(
        teachers.map(async (teacher) => {
          const lastMsg = await Message.findOne({
            $or: [
              {
                senderId: teacher._id,
                senderType: 'Employee',
                receiverId: adminObjectId,
                receiverType: 'Admin',
              },
              {
                receiverId: teacher._id,
                receiverType: 'Employee',
                senderId: adminObjectId,
                senderType: 'Admin',
              },
            ],
          })
            .sort({ timestamp: -1 })
            .lean();

          return {
            _id: teacher._id,
            name: teacher.name || teacher.email || 'Unknown Teacher',
            role: 'teacher',
            lastMessage: lastMsg?.attachment ? '📎 Attachment' : (lastMsg?.content || ''),
            timestamp: lastMsg?.timestamp || null,
          };
        })
      );

      res.json(enrichedTeachers);
    } catch (err) {
      console.error('Error in /employee route:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Get all students
  router.get('/student', isAdmin, async (req, res) => {
    const { adminId } = req.query;

    if (!adminId) {
      return res.status(400).json({ error: 'adminId is required' });
    }

    const adminObjectId = new mongoose.Types.ObjectId(adminId);

    try {
      // Filter active and non-deleted students
      const students = await Student.find({ isActive: true, isDeleted: false })
        .select('_id studentName email')
        .lean();

      console.log('Found students:', students.length);

      const enrichedStudents = await Promise.all(
        students.map(async (student) => {
          const lastMsg = await Message.findOne({
            $or: [
              {
                senderId: student._id,
                senderType: 'Student',
                receiverId: adminObjectId,
                receiverType: 'Admin',
              },
              {
                receiverId: student._id,
                receiverType: 'Student',
                senderId: adminObjectId,
                senderType: 'Admin',
              },
            ],
          })
            .sort({ timestamp: -1 })
            .lean();

          return {
            _id: student._id,
            name: student.studentName || student.email || 'Unknown Student',
            role: 'student',
            lastMessage: lastMsg?.attachment ? '📎 Attachment' : (lastMsg?.content || ''),
            timestamp: lastMsg?.timestamp || null,
          };
        })
      );

      res.json(enrichedStudents);
    } catch (err) {
      console.error('Error in /student route:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Get messages by conversationId
  router.get('/messages/:conversationId', async (req, res) => {
    const { conversationId } = req.params;

    try {
      const messages = await Message.find({ conversationId }).sort({ timestamp: 1 });
      res.json(messages);
    } catch (err) {
      res.status(500).json({ error: err.message });
    }
  });

  // Get all conversations (Teacher-Student) with participants' names
  router.get('/all-chats', async (req, res) => {
    try {
      const allChats = await Message.aggregate([
        {
          $match: {
            receiverType: { $ne: 'Admin' },
          },
        },
        {
          $group: {
            _id: { conversationId: '$conversationId' },
            messages: { $push: '$$ROOT' },
            senderId: { $first: '$senderId' },
            receiverId: { $first: '$receiverId' },
          },
        },
        {
          $lookup: {
            from: 'employees',
            localField: 'senderId',
            foreignField: '_id',
            as: 'senderEmployee',
          },
        },
        {
          $lookup: {
            from: 'students',
            localField: 'senderId',
            foreignField: '_id',
            as: 'senderStudent',
          },
        },
        {
          $lookup: {
            from: 'employees',
            localField: 'receiverId',
            foreignField: '_id',
            as: 'receiverEmployee',
          },
        },
        {
          $lookup: {
            from: 'students',
            localField: 'receiverId',
            foreignField: '_id',
            as: 'receiverStudent',
          },
        },
        {
          $project: {
            conversationId: '$_id.conversationId',
            messages: 1,
            teacherName: {
              $cond: {
                if: { $gt: [{ $size: '$senderEmployee' }, 0] },
                then: { $arrayElemAt: ['$senderEmployee.employeeName', 0] },
                else: { $arrayElemAt: ['$receiverEmployee.employeeName', 0] },
              },
            },
            studentName: {
              $cond: {
                if: { $gt: [{ $size: '$senderStudent' }, 0] },
                then: { $arrayElemAt: ['$senderStudent.studentName', 0] },
                else: { $arrayElemAt: ['$receiverStudent.studentName', 0] },
              },
            },
          },
        },
        {
          $match: {
            teacherName: { $ne: null },
            studentName: { $ne: null },
          },
        },
      ]);
      res.json(allChats);
    } catch (err) {
      res.status(500).json({ error: err.message });
    }
  });

  // Send a message
  router.post('/messages', upload.single('attachment'), async (req, res) => {
    try {
      const { senderId, receiverId, receiverType, content } = req.body;

      if (!receiverId || !receiverType || !content) {
        return res.status(400).json({ error: 'Missing required fields' });
      }
      const admin = await Admin.findById(senderId);
      if (!admin) return res.status(404).json({ error: 'Admin not found' });

      // Convert 'Employee' to 'Teacher' for conversation model compatibility
      const normalizedReceiverType = receiverType === 'Employee' ? 'Teacher' : receiverType;

      // Validate receiver (Admin or Student or Teacher)
      let receiver;
      if (receiverType === 'Employee' || receiverType === 'Teacher') {
        receiver = await Employee.findById(receiverId);
      } else if (receiverType === 'Student') {
        receiver = await Student.findById(receiverId);
      }

      if (!receiver) return res.status(404).json({ error: 'Receiver not found' });
      const senderType = 'Admin';

      // Find or create a conversation using normalized types
      let conversation = await Conversation.findOne({
        participants: {
          $all: [
            { userId: senderId, userType: senderType },
            { userId: receiverId, userType: normalizedReceiverType },
          ],
        },
      });

      if (!conversation) {
        conversation = new Conversation({
          participants: [
            { userId: senderId, userType: senderType },
            { userId: receiverId, userType: normalizedReceiverType },
          ],
          isGroup: false,
        });
        await conversation.save();
        console.log('Created new conversation:', conversation._id);
      }

      // Prepare message data
      const messageData = {
        conversationId: conversation._id,
        senderId,
        senderType,
        receiverId,
        receiverType: normalizedReceiverType,
        content,
        timestamp: new Date(),
      };

      // Add attachment if file was uploaded (use consistent schema)
      if (req.file) {
        messageData.attachment = {
          fileName: req.file.originalname,
          fileUrl: `/uploads/chat/${req.file.filename}`,
          fileType: req.file.mimetype,
          fileSize: req.file.size,
        };
        console.log('📎 File uploaded:', messageData.attachment);
      }

      // Create a new message
      const message = new Message(messageData);

      await message.save();

      console.log('💾 Message saved with attachment:', {
        id: message._id,
        hasAttachment: !!message.attachment,
        attachmentType: message.attachment?.type,
        attachmentUrl: message.attachment?.url
      });

      // Conversation timestamp is automatically updated by Message post-save middleware

      // Emit socket event for real-time delivery
      const conversationId = conversation._id.toString();

      console.log('🔔 Emitting admin message to conversation:', conversationId);
      console.log('🔔 Message data:', message);
      console.log('🔔 Sender ID (Admin):', senderId);
      console.log('🔔 Receiver ID:', receiverId);
      console.log('🔔 Online users:', Object.keys(users));

      // Emit to conversation room
      io.to(conversationId).emit('receive_message', message);
      console.log(`📡 Emitted to conversation room: ${conversationId}`);

      // Emit to receiver's sockets
      const receiverIdStr = receiverId.toString();
      if (users[receiverIdStr]) {
        users[receiverIdStr].forEach((socketId) => {
          io.to(socketId).emit('receive_message', message);
          console.log(`📡 Emitted to receiver socket: ${socketId}`);
        });
        console.log(`📨 Admin message delivered to receiver ${receiverId} in real-time`);
      } else {
        console.log(`⚠️ Receiver ${receiverId} is offline. Online users:`, Object.keys(users));
      }

      // Also emit to sender (admin) sockets so they see their own message
      const senderIdStr = senderId.toString();
      if (users[senderIdStr]) {
        users[senderIdStr].forEach((socketId) => {
          io.to(socketId).emit('receive_message', message);
          console.log(`📡 Emitted to sender (admin) socket: ${socketId}`);
        });
        console.log(`📨 Admin message delivered to admin (sender) in real-time`);
      } else {
        console.log(`⚠️ Admin ${senderId} is offline. Online users:`, Object.keys(users));
      }

      // Send notification to receiver
      try {
        const { createNotification, NotificationTemplates } = await import('../../utils/notificationHelper.js');
        const notificationData = NotificationTemplates.messageReceived('Admin');

        await createNotification({
          userId: receiverId,
          userType: normalizedReceiverType.toLowerCase(),
          type: notificationData.type,
          title: notificationData.title,
          message: notificationData.message,
          relatedId: message._id,
          relatedModel: 'Message',
          priority: notificationData.priority,
          actionUrl: normalizedReceiverType === 'Teacher' ? '/teacher/chat' : '/student/chat',
          io
        });

        console.log(`📢 Sent message notification to ${normalizedReceiverType} ${receiverId}`);
      } catch (notificationError) {
        console.error('Failed to send message notification:', notificationError);
        // Don't fail the request if notifications fail
      }

      res.status(200).json({ message: 'Message sent successfully!', data: message });
    } catch (err) {
      console.error('Error sending message:', err);
      res.status(500).json({ error: err.message });
    }
  });

  router.post('/broadcast/student', upload.single('attachment'), async (req, res) => {
    try {
      const { senderId, content } = req.body;
      const senderType = 'Admin';

      // Prepare message data
      const messageData = {
        conversationId: 'broadcast-students',
        senderId,
        senderType,
        receiverId: null, // No specific receiver, it's a broadcast
        receiverType: 'Student',
        content,
        isGroupMessage: true,
        timestamp: new Date(),
      };

      // Add attachment if file was uploaded (use consistent schema)
      if (req.file) {
        messageData.attachment = {
          fileName: req.file.originalname,
          fileUrl: `/uploads/chat/${req.file.filename}`,
          fileType: req.file.mimetype,
          fileSize: req.file.size,
        };
      }

      // Save a single broadcast message
      const broadcastMessage = new Message(messageData);
      await broadcastMessage.save();

      // Emit broadcast message to all students in real-time
      io.to('students').emit('receive_broadcast', broadcastMessage);
      console.log('📢 Broadcast message sent to all students');

      res.status(200).json({ message: 'Broadcast message sent to students successfully!' });
    } catch (err) {
      console.error('Error sending broadcast message:', err);
      res.status(500).json({ error: err.message });
    }
  });

  router.post('/broadcast/employee', upload.single('attachment'), async (req, res) => {
    try {
      const { senderId, content } = req.body;
      const senderType = 'Admin';

      // Prepare message data
      const messageData = {
        conversationId: 'broadcast-teachers',
        senderId,
        senderType,
        receiverId: null, // No specific receiver, it's a broadcast
        receiverType: 'Teacher',
        content,
        isGroupMessage: true,
        timestamp: new Date(),
      };

      // Add attachment if file was uploaded (use consistent schema)
      if (req.file) {
        messageData.attachment = {
          fileName: req.file.originalname,
          fileUrl: `/uploads/chat/${req.file.filename}`,
          fileType: req.file.mimetype,
          fileSize: req.file.size,
        };
      }

      // Save a single broadcast message - use 'Teacher' instead of 'Employee'
      const broadcastMessage = new Message(messageData);
      await broadcastMessage.save();

      // Emit broadcast message to all teachers in real-time
      io.to('teachers').emit('receive_broadcast', broadcastMessage);
      console.log('📢 Broadcast message sent to all teachers');

      res.status(200).json({ message: 'Broadcast message sent to teachers successfully!' });
    } catch (err) {
      console.error('Error sending broadcast message:', err);
      res.status(500).json({ error: err.message });
    }
  });
  function getModelForUserType(userType) {
    const modelMap = {
      Admin: Admin,
      Employee: Employee,
      Teacher: Employee, // Map 'Teacher' to Employee model
      Student: Student,
    };
    return modelMap[userType] || null; // Ensure correct model mapping
  }

  router.get('/conversation', async (req, res) => {
    try {
      const { adminId } = req.query;
      if (!adminId) return res.status(400).json({ error: 'Admin ID is required' });

      // Fetch conversations that are visible to the admin
      const conversations = await Conversation.find({ isVisibleToAdmin: true }).sort({ timestamp: -1 });

      if (!conversations || conversations.length === 0) {
        return res.status(404).json({ message: 'No conversation found' });
      }

      // Fetch messages and manually populate senderId & receiverId
      const messages = await Message.find({
        conversationId: { $in: conversations.map((c) => c._id) },
      }).sort({ timestamp: -1 });

      // **Manually Populate senderId and receiverId**
      const populatedMessages = await Promise.all(
        messages.map(async (msg) => {
          const senderModel = getModelForUserType(msg.senderType);
          const receiverModel = getModelForUserType(msg.receiverType);

          if (senderModel) {
            msg.senderId = await senderModel.findById(msg.senderId).select('employeeName studentName');
          }

          if (receiverModel) {
            msg.receiverId = await receiverModel.findById(msg.receiverId).select('employeeName studentName');
          }

          return msg;
        })
      );

      res.status(200).json({ messages: populatedMessages });
    } catch (err) {
      console.error('Error fetching messages:', err);
      res.status(500).json({ error: err.message });
    }
  });

  router.get('/admin-individualMessage', async (req, res) => {
    try {
      const { adminId, receiverId } = req.query;

      if (!adminId || !receiverId) {
        return res.status(400).json({ error: 'Admin ID and Receiver ID are required' });
      }

      // Fetch messages where admin is either sender or receiver
      const individualMessages = await Message.find({
        $or: [
          { senderId: adminId, senderType: 'Admin', receiverId: receiverId }, // Admin sent message
          { senderId: receiverId, receiverType: 'Admin', receiverId: adminId }, // Admin received message
        ],
        isGroupMessage: false,
      }).sort({ timestamp: 1 }); // Sort in ascending order for proper chat flow

      res.status(200).json({ messages: individualMessages });
    } catch (err) {
      console.error('Error fetching individual messages:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // router.get('/broadcast/students', async (req, res) => {
  //   try {
  //     console.log('broadcast student>>')

  //     const messages = await Message.find({
  //       receiverType: 'Student',
  //       isGroupMessage: true,
  //     }).sort({ timestamp: -1 });

  //     if (!messages.length) {
  //       return res.status(404).json({ error: 'No broadcast messages found for students' });
  //     }

  //     res.status(200).json({ messages });
  //   } catch (err) {
  //     console.error('Error fetching student broadcast messages:', err);
  //     res.status(500).json({ error: err.message });
  //   }
  // });

  // router.get('/broadcast/teachers', async (req, res) => {
  //   try {
  //     console.log('broadcast teacher>>')
  //     const messages = await Message.find({
  //       receiverType: 'Employee',
  //       isGroupMessage: true,
  //     }).sort({ timestamp: -1 });

  //     if (!messages.length) {
  //       return res.status(404).json({ error: 'No broadcast messages found for teachers' });
  //     }

  //     res.status(200).json({ messages });
  //   } catch (err) {
  //     console.error('Error fetching teacher broadcast messages:', err);
  //     res.status(500).json({ error: err.message });
  //   }
  // });
  router.get('/broadcast/:type', async (req, res) => {
    try {
      const { type } = req.params;
      let receiverType;
      if (type === 'students') receiverType = 'Student';
      else if (type === 'teachers') receiverType = 'Teacher'; // Changed from 'Employee' to 'Teacher'
      else return res.status(400).json({ error: 'Invalid broadcast type' });

      const messages = await Message.find({
        conversationId: `broadcast-${type}`,
        isGroupMessage: true,
        receiverType,
      }).sort({ timestamp: 1 });

      res.status(200).json({ messages });
    } catch (err) {
      console.error('Error fetching broadcast messages:', err);
      res.status(500).json({ error: err.message });
    }
  });

  return router;
}
