const SecurityHeaders = require('./securityHeaders');
const rateLimiter = require('./rateLimiter');
const validation = require('./validation');
const enterpriseAuth = require('./enterpriseAuth');
const rbac = require('./rbac');
const auditLogger = require('../utils/auditLogger');
const encryption = require('../utils/encryption');

class EnterpriseSecurityPipeline {
  constructor() {
    this.initializeHealthCheck();
  }

  // Initialize health monitoring
  initializeHealthCheck() {
    this.healthStats = {
      requests: 0,
      errors: 0,
      authFailures: 0,
      rateLimitHits: 0,
      lastReset: Date.now()
    };

    // Reset stats every hour
    setInterval(() => {
      this.resetHealthStats();
    }, 60 * 60 * 1000);
  }

  resetHealthStats() {
    this.healthStats = {
      requests: 0,
      errors: 0,
      authFailures: 0,
      rateLimitHits: 0,
      lastReset: Date.now()
    };
  }

  // Security middleware pipeline for different contexts
  getSecurityPipeline(context = 'api') {
    const pipeline = [];

    switch (context) {
      case 'auth':
        return this.getAuthPipeline();
      case 'api':
        return this.getAPIPipeline();
      case 'upload':
        return this.getUploadPipeline();
      case 'public':
        return this.getPublicPipeline();
      case 'admin':
        return this.getAdminPipeline();
      default:
        return this.getDefaultPipeline();
    }
  }

  // Authentication endpoints pipeline
  getAuthPipeline() {
    return [
      // Basic security headers
      ...SecurityHeaders.getAllMiddleware(),
      
      // Rate limiting for auth (stricter)
      rateLimiter.getMiddleware('auth'),
      
      // Request tracking
      this.requestTracker(),
      
      // Input sanitization
      validation.sanitizeInput(),
      
      // Audit logging
      auditLogger.createRequestLogger()
    ];
  }

  // API endpoints pipeline
  getAPIPipeline() {
    return [
      // Security headers
      ...SecurityHeaders.getAllMiddleware(),
      
      // DDoS protection
      rateLimiter.getMiddleware('ddos'),
      
      // General rate limiting
      rateLimiter.getMiddleware('api'),
      
      // Progressive slow down
      rateLimiter.getMiddleware('slowdown'),
      
      // Request tracking
      this.requestTracker(),
      
      // Authentication
      enterpriseAuth.authenticate,
      
      // Input sanitization and validation
      validation.sanitizeInput(),
      
      // Audit logging
      auditLogger.createRequestLogger(),
      
      // Security monitoring
      this.securityMonitor()
    ];
  }

  // File upload pipeline
  getUploadPipeline() {
    return [
      // Security headers
      ...SecurityHeaders.getAllMiddleware(),
      
      // Stricter rate limiting for uploads
      rateLimiter.getMiddleware('auth'),
      
      // Request tracking
      this.requestTracker(),
      
      // Authentication required
      enterpriseAuth.authenticate,
      
      // File upload validation
      this.fileUploadSecurity(),
      
      // Audit logging
      auditLogger.createRequestLogger()
    ];
  }

  // Public endpoints pipeline (lighter security)
  getPublicPipeline() {
    return [
      // Basic security headers
      SecurityHeaders.customSecurityHeaders(),
      SecurityHeaders.configureCORS(),
      
      // Basic rate limiting
      rateLimiter.getMiddleware('general'),
      
      // Request tracking
      this.requestTracker(),
      
      // Input sanitization
      validation.sanitizeInput(),
      
      // Basic audit logging
      auditLogger.createRequestLogger()
    ];
  }

  // Admin endpoints pipeline (maximum security)
  getAdminPipeline() {
    return [
      // Full security headers
      ...SecurityHeaders.getAllMiddleware(),
      
      // Strict DDoS protection
      rateLimiter.getMiddleware('ddos'),
      
      // Strict rate limiting
      rateLimiter.getMiddleware('auth'),
      
      // Request tracking
      this.requestTracker(),
      
      // Authentication required
      enterpriseAuth.authenticate,
      
      // Admin role required
      rbac.requirePermission('system.settings'),
      
      // Enhanced input validation
      validation.sanitizeInput(),
      validation.validateBusinessRules(),
      
      // Full audit logging
      auditLogger.createRequestLogger(),
      
      // Enhanced security monitoring
      this.enhancedSecurityMonitor()
    ];
  }

  // Default pipeline
  getDefaultPipeline() {
    return [
      // Security headers
      ...SecurityHeaders.getAllMiddleware(),
      
      // Rate limiting
      rateLimiter.getMiddleware('general'),
      
      // Request tracking
      this.requestTracker(),
      
      // Input sanitization
      validation.sanitizeInput(),
      
      // Audit logging
      auditLogger.createRequestLogger()
    ];
  }

  // Request tracking middleware
  requestTracker() {
    return (req, res, next) => {
      this.healthStats.requests++;
      
      // Add unique request ID
      req.requestId = require('crypto').randomUUID();
      res.setHeader('X-Request-ID', req.requestId);
      
      // Track request timing
      req.startTime = Date.now();
      
      // Monitor response
      res.on('finish', () => {
        const duration = Date.now() - req.startTime;
        
        // Log slow requests
        if (duration > 5000) {
          auditLogger.warn('Slow request detected', {
            requestId: req.requestId,
            method: req.method,
            path: req.path,
            duration: `${duration}ms`,
            userId: req.user?.id
          });
        }
        
        // Track errors
        if (res.statusCode >= 400) {
          this.healthStats.errors++;
          
          if (res.statusCode === 401 || res.statusCode === 403) {
            this.healthStats.authFailures++;
          }
          
          if (res.statusCode === 429) {
            this.healthStats.rateLimitHits++;
          }
        }
      });
      
      next();
    };
  }

  // Security monitoring middleware
  securityMonitor() {
    return (req, res, next) => {
      // Check for suspicious patterns
      const suspiciousPatterns = [
        /\b(union|select|insert|update|delete|drop|create|alter)\b/i,
        /<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi,
        /javascript:/i,
        /on\w+\s*=/i,
        /\b(eval|exec|system|shell_exec)\b/i
      ];

      const requestBody = JSON.stringify(req.body || {});
      const requestQuery = JSON.stringify(req.query || {});
      const allInput = requestBody + requestQuery + req.path;

      for (const pattern of suspiciousPatterns) {
        if (pattern.test(allInput)) {
          auditLogger.logSecurityEvent('SUSPICIOUS_INPUT_DETECTED', 'high', {
            pattern: pattern.source,
            path: req.path,
            method: req.method,
            ip: req.ip,
            userAgent: req.get('User-Agent'),
            userId: req.user?.id,
            input: encryption.maskSensitiveData({ body: req.body, query: req.query })
          });

          // In production, you might want to block these requests
          if (process.env.NODE_ENV === 'production' && process.env.BLOCK_SUSPICIOUS_REQUESTS === 'true') {
            return res.status(400).json({
              error: 'Request contains suspicious content',
              code: 'SUSPICIOUS_INPUT'
            });
          }
        }
      }

      next();
    };
  }

  // Enhanced security monitoring for admin endpoints
  enhancedSecurityMonitor() {
    return (req, res, next) => {
      // Run standard security monitoring
      this.securityMonitor()(req, res, () => {
        // Additional admin-specific checks
        
        // Log all admin actions
        auditLogger.logSecurityEvent('ADMIN_ACTION', 'info', {
          action: `${req.method} ${req.path}`,
          userId: req.user?.id,
          ip: req.ip,
          userAgent: req.get('User-Agent'),
          timestamp: new Date().toISOString()
        });

        // Check for privilege escalation attempts
        if (req.body?.userType || req.body?.role) {
          auditLogger.logSecurityEvent('PRIVILEGE_CHANGE_ATTEMPT', 'warn', {
            targetUserType: req.body.userType || req.body.role,
            adminUserId: req.user?.id,
            ip: req.ip,
            path: req.path
          });
        }

        next();
      });
    };
  }

  // File upload security
  fileUploadSecurity() {
    return (req, res, next) => {
      const multer = require('multer');
      const path = require('path');
      const crypto = require('crypto');

      // Configure multer with security settings
      const storage = multer.diskStorage({
        destination: (req, file, cb) => {
          cb(null, 'uploads/');
        },
        filename: (req, file, cb) => {
          // Generate secure filename
          const ext = path.extname(file.originalname);
          const filename = crypto.randomUUID() + ext;
          cb(null, filename);
        }
      });

      const fileFilter = (req, file, cb) => {
        // Check file type
        const allowedMimeTypes = [
          'image/jpeg',
          'image/png',
          'image/gif',
          'application/pdf',
          'text/plain'
        ];

        if (!allowedMimeTypes.includes(file.mimetype)) {
          const error = new Error('File type not allowed');
          error.code = 'INVALID_FILE_TYPE';
          return cb(error, false);
        }

        // Check file extension
        const ext = path.extname(file.originalname).toLowerCase();
        const allowedExtensions = ['.jpg', '.jpeg', '.png', '.gif', '.pdf', '.txt'];
        
        if (!allowedExtensions.includes(ext)) {
          const error = new Error('File extension not allowed');
          error.code = 'INVALID_FILE_EXTENSION';
          return cb(error, false);
        }

        cb(null, true);
      };

      const upload = multer({
        storage,
        fileFilter,
        limits: {
          fileSize: 10 * 1024 * 1024, // 10MB
          files: 5 // Maximum 5 files
        }
      });

      // Apply multer middleware
      upload.array('files')(req, res, (err) => {
        if (err) {
          auditLogger.logSecurityEvent('FILE_UPLOAD_REJECTED', 'warn', {
            error: err.message,
            code: err.code,
            userId: req.user?.id,
            ip: req.ip
          });

          return res.status(400).json({
            error: 'File upload failed',
            reason: err.message,
            code: err.code
          });
        }

        // Log successful upload
        if (req.files && req.files.length > 0) {
          auditLogger.logSecurityEvent('FILE_UPLOAD_SUCCESS', 'info', {
            fileCount: req.files.length,
            files: req.files.map(f => ({
              originalName: f.originalname,
              size: f.size,
              mimetype: f.mimetype
            })),
            userId: req.user?.id,
            ip: req.ip
          });
        }

        next();
      });
    };
  }

  // Error handling middleware
  errorHandler() {
    return (err, req, res, next) => {
      this.healthStats.errors++;

      // Log the error
      auditLogger.error('Unhandled error', {
        error: {
          message: err.message,
          stack: err.stack,
          name: err.name
        },
        requestId: req.requestId,
        path: req.path,
        method: req.method,
        userId: req.user?.id,
        ip: req.ip
      });

      // Don't expose internal errors in production
      const isDevelopment = process.env.NODE_ENV === 'development';
      
      let statusCode = err.status || err.statusCode || 500;
      let message = 'Internal server error';
      let details = null;

      // Handle specific error types
      if (err.name === 'ValidationError') {
        statusCode = 400;
        message = 'Validation failed';
        details = isDevelopment ? err.details : null;
      } else if (err.name === 'UnauthorizedError') {
        statusCode = 401;
        message = 'Unauthorized';
      } else if (err.code === 'PERMISSION_DENIED') {
        statusCode = 403;
        message = 'Access denied';
      } else if (err.code === 'RATE_LIMIT_EXCEEDED') {
        statusCode = 429;
        message = 'Too many requests';
      } else if (isDevelopment) {
        message = err.message;
        details = err.stack;
      }

      res.status(statusCode).json({
        error: message,
        code: err.code || 'INTERNAL_ERROR',
        requestId: req.requestId,
        timestamp: new Date().toISOString(),
        ...(details && { details })
      });
    };
  }

  // Health check endpoint
  healthCheck() {
    return (req, res) => {
      const uptime = process.uptime();
      const memoryUsage = process.memoryUsage();
      
      const health = {
        status: 'healthy',
        timestamp: new Date().toISOString(),
        uptime: `${Math.floor(uptime / 60)} minutes`,
        memory: {
          used: Math.round(memoryUsage.heapUsed / 1024 / 1024) + ' MB',
          total: Math.round(memoryUsage.heapTotal / 1024 / 1024) + ' MB'
        },
        stats: {
          ...this.healthStats,
          errorRate: this.healthStats.requests > 0 
            ? ((this.healthStats.errors / this.healthStats.requests) * 100).toFixed(2) + '%'
            : '0%'
        }
      };

      // Check for unhealthy conditions
      if (this.healthStats.errorRate > 10) {
        health.status = 'degraded';
        health.warnings = ['High error rate detected'];
      }

      auditLogger.logHealthCheck(health.status, health);

      res.json(health);
    };
  }

  // Security configuration endpoint (admin only)
  getSecurityConfig() {
    return [
      ...this.getAdminPipeline(),
      (req, res) => {
        const config = {
          rateLimiting: {
            enabled: true,
            windows: ['15 minutes'],
            limits: {
              general: 1000,
              api: 100,
              auth: 5
            }
          },
          authentication: {
            jwtExpiry: '15m',
            refreshTokenExpiry: '7d',
            algorithm: 'HS256'
          },
          encryption: {
            algorithm: 'aes-256-gcm',
            keyRotation: 'monthly'
          },
          audit: {
            retention: '90 days',
            level: 'info'
          },
          cors: {
            enabled: true,
            origins: ['localhost:3001', 'localhost:3002', 'localhost:3003']
          }
        };

        res.json(config);
      }
    ];
  }
}

// Create singleton instance
const securityPipeline = new EnterpriseSecurityPipeline();

module.exports = securityPipeline;