import express from 'express';
import Chat from '../../models/admin/Chat.js';
import Student from '../../models/student/Student.js';
import Teacher from '../../models/teacher/Teacher.js';
import { auth, adminAuth } from '../../middleware/auth.js';
import { sendMessage } from '../../services/messageService.js';

const router = express.Router();

router.post('/start', auth, async (req, res) => {
  try {
    console.log('💬 Chat start request:', { 
      body: req.body, 
      user: { id: req.user._id, type: req.user.userType } 
    });

    const { recipientId, recipientType, subject } = req.body;
    const senderId = req.user._id;
    const senderType = req.user.userType;

    // Validate required fields
    if (!recipientId || !recipientType) {
      return res.status(400).json({ message: 'recipientId and recipientType are required' });
    }

    // Get the actual User IDs for participants
    let senderUserId = senderId;
    let recipientUserId = null;

    // Validate recipient exists and get their user ID
    if (recipientType === 'student') {
      const student = await Student.findById(recipientId);
      if (!student) {
        console.log('❌ Student not found:', recipientId);
        return res.status(404).json({ message: 'Student not found' });
      }
      recipientUserId = student.userId;
      console.log('👨‍🎓 Student found:', { studentId: recipientId, userId: recipientUserId });
    } else if (recipientType === 'teacher') {
      const teacher = await Teacher.findById(recipientId);
      if (!teacher) {
        console.log('❌ Teacher not found:', recipientId);
        return res.status(404).json({ message: 'Teacher not found' });
      }
      recipientUserId = teacher.userId;
      console.log('👨‍🏫 Teacher found:', { teacherId: recipientId, userId: recipientUserId });
    } else if (recipientType === 'admin') {
      // For admin recipient, recipientId should be the user ID
      recipientUserId = recipientId;
    } else {
      return res.status(400).json({ message: 'Invalid recipient type' });
    }

    if (!recipientUserId) {
      return res.status(400).json({ message: 'Could not resolve recipient user ID' });
    }

    // Student validation for chat permissions
    if (senderType === 'student') {
      if (!senderId.toString().startsWith('dev-')) {
        const student = await Student.findOne({ userId: senderId });
        if (!student) {
          return res.status(404).json({ message: 'Student profile not found' });
        }

        if (recipientType === 'teacher') {
          const isAssigned = student.assignedTeachers.some(
            t => t.teacherId.toString() === recipientId && t.subject === subject
          );
          if (!isAssigned) {
            return res.status(403).json({ message: 'You can only chat with assigned teachers for specific subjects' });
          }
        }
      }
    }

    console.log('🔍 Looking for existing chat between:', { senderUserId, recipientUserId });

    // Check for existing chat
    let existingChat = await Chat.findOne({
      'participants.userId': { $all: [senderUserId, recipientUserId] },
      ...(subject && { subject })
    });

    if (existingChat) {
      console.log('✅ Found existing chat:', existingChat._id);
      return res.json({ chat: existingChat });
    }

    // Determine chat type correctly
    const chatType = senderType === 'admin' && recipientType === 'student' ? 'student-admin' :
                     senderType === 'admin' && recipientType === 'teacher' ? 'teacher-admin' :
                     senderType === 'student' && recipientType === 'teacher' ? 'student-teacher' :
                     senderType === 'student' && recipientType === 'admin' ? 'student-admin' :
                     senderType === 'teacher' && recipientType === 'admin' ? 'teacher-admin' :
                     'student-teacher';

    console.log('📝 Creating new chat:', { 
      chatType, 
      senderType, 
      recipientType, 
      senderUserId, 
      recipientUserId 
    });

    const newChat = new Chat({
      participants: [
        { userId: senderUserId, userType: senderType },
        { userId: recipientUserId, userType: recipientType }
      ],
      chatType,
      subject,
      messages: []
    });

    await newChat.save();
    console.log('✅ Chat created successfully:', newChat._id);
    
    res.status(201).json({ chat: newChat });
  } catch (error) {
    console.error('❌ Start chat error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

// router.post('/:chatId/message', auth, async (req, res) => {
//   try {
//     const { message } = req.body;
//     const senderId = req.user._id;
//     const senderType = req.user.userType;

//     const chat = await Chat.findById(req.params.chatId);
//     if (!chat) {
//       return res.status(404).json({ message: 'Chat not found' });
//     }

//     const isParticipant = chat.participants.some(p => p.userId.toString() === senderId.toString());
//     if (!isParticipant) {
//       return res.status(403).json({ message: 'Access denied' });
//     }

//     chat.messages.push({
//       senderId,
//       senderType,
//       message,
//       timestamp: new Date()
//     });

//     chat.lastActivity = new Date();
//     await chat.save();

//     res.json({ message: 'Message sent successfully', chat });
//   } catch (error) {
//     console.error('Send message error:', error);
//     res.status(500).json({ message: 'Server error' });
//   }
// });

router.post('/:chatId/message', auth, async (req, res) => {
  try {
    const { message } = req.body;
    const senderId = req.user._id;
    const senderType = req.user.userType;

    const io = req.app.get('io'); // retrieve socket.io instance from app

    const { newMsg, chat } = await sendMessage({
      chatId: req.params.chatId,
      senderId,
      senderType,
      message,
      io
    });

    res.json({
      message: 'Message sent successfully',
      newMessage: newMsg,
      chatId: chat._id
    });
  } catch (error) {
    console.error('Send message error:', error);
    res.status(500).json({ message: error.message || 'Server error' });
  }
});


// router.get('/my-chats', auth, async (req, res) => {
//   try {
//     const userId = req.user._id;
//     const userType = req.user.userType;
    
//     console.log('📋 My chats request:', { userId, userType });
    
//     // Development mode: Return mock data
//     if (userId.toString().startsWith('dev-')) {
//       return res.json({
//         chats: [
//           {
//             _id: 'chat1',
//             subject: 'Mathematics',
//             lastActivity: new Date(),
//             otherParticipant: {
//               name: 'Mr. Smith',
//               type: 'teacher'
//             },
//             unreadCount: 2
//           },
//           {
//             _id: 'chat2',
//             subject: 'Physics',
//             lastActivity: new Date(Date.now() - 24 * 60 * 60 * 1000),
//             otherParticipant: {
//               name: 'Ms. Johnson',
//               type: 'teacher'
//             },
//             unreadCount: 0
//           }
//         ]
//       });
//     }
    
//     const chats = await Chat.find({
//       'participants.userId': userId,
//       isActive: true
//     })
//     .populate('participants.userId', 'username')
//     .sort({ lastActivity: -1 });

//     console.log(`💬 Found ${chats.length} chats for user ${userId}`);

//     const chatsWithDetails = await Promise.all(
//       chats.map(async (chat) => {
//         try {
//           const otherParticipant = chat.participants.find(p => 
//             p.userId && p.userId._id && p.userId._id.toString() !== userId.toString()
//           );
          
//           let participantDetails = { name: 'Unknown User', type: 'unknown' };

//           if (otherParticipant && otherParticipant.userId) {
//             if (otherParticipant.userType === 'student') {
//               if (otherParticipant.userId._id.toString().startsWith('dev-')) {
//                 participantDetails = { name: 'Development Student', type: 'student' };
//               } else {
//                 const student = await Student.findOne({ userId: otherParticipant.userId._id });
//                 if (student) {
//                   participantDetails = { 
//                     name: student.studentName, 
//                     type: 'student',
//                     email: student.email 
//                   };
//                 }
//               }
//             } else if (otherParticipant.userType === 'teacher') {
//               if (otherParticipant.userId._id.toString().startsWith('dev-')) {
//                 participantDetails = { name: 'Development Teacher', type: 'teacher' };
//               } else {
//                 const teacher = await Teacher.findOne({ userId: otherParticipant.userId._id });
//                 if (teacher) {
//                   participantDetails = { 
//                     name: teacher.name, 
//                     type: 'teacher',
//                     email: teacher.email 
//                   };
//                 }
//               }
//             } else if (otherParticipant.userType === 'admin') {
//               participantDetails = { name: 'Admin', type: 'admin' };
//             }
//           }

//           const unreadCount = chat.messages.filter(m => 
//             m.senderId && m.senderId.toString() !== userId.toString() && 
//             !m.readBy.some(r => r.userId && r.userId.toString() === userId.toString())
//           ).length;

//           return {
//             _id: chat._id,
//             subject: chat.subject,
//             chatType: chat.chatType,
//             lastActivity: chat.lastActivity,
//             otherParticipant: participantDetails,
//             unreadCount,
//             messageCount: chat.messages.length,
//             lastMessage: chat.messages.length > 0 ? {
//               text: chat.messages[chat.messages.length - 1].message,
//               timestamp: chat.messages[chat.messages.length - 1].timestamp,
//               senderId: chat.messages[chat.messages.length - 1].senderId
//             } : null
//           };
//         } catch (chatError) {
//           console.error('Error processing chat:', chat._id, chatError);
//           return {
//             _id: chat._id,
//             subject: chat.subject,
//             chatType: chat.chatType,
//             lastActivity: chat.lastActivity,
//             otherParticipant: { name: 'Error Loading User', type: 'unknown' },
//             unreadCount: 0,
//             messageCount: 0,
//             lastMessage: null
//           };
//         }
//       })
//     );

//     console.log(`✅ Processed ${chatsWithDetails.length} chats successfully`);
//     res.json({ chats: chatsWithDetails });
//   } catch (error) {
//     console.error('❌ Get chats error:', error);
//     res.status(500).json({ message: 'Server error', error: error.message });
//   }
// });

// router.get('/:chatId/messages', auth, async (req, res) => {
//   try {
//     const userId = req.user._id;
//     const { page = 1, limit = 50 } = req.query;

//     const chat = await Chat.findById(req.params.chatId);
//     if (!chat) {
//       return res.status(404).json({ message: 'Chat not found' });
//     }

//     const isParticipant = chat.participants.some(p => p.userId.toString() === userId.toString());
//     if (!isParticipant) {
//       return res.status(403).json({ message: 'Access denied' });
//     }

//     const totalMessages = chat.messages.length;
//     const startIndex = Math.max(0, totalMessages - (page * limit));
//     const endIndex = totalMessages - ((page - 1) * limit);
    
//     const messages = chat.messages.slice(startIndex, endIndex);

//     // Mark messages as read by this user
//     let hasUpdates = false;
//     chat.messages.forEach(msg => {
//       if (msg.senderId.toString() !== userId.toString()) {
//         const alreadyRead = msg.readBy.some(r => r.userId.toString() === userId.toString());
//         if (!alreadyRead) {
//           msg.readBy.push({ userId: userId, readAt: new Date() });
//           hasUpdates = true;
//         }
//       }
//     });
    
//     if (hasUpdates) {
//       await chat.save();
//     }

//     // Transform messages to match frontend expectations
//     const transformedMessages = await Promise.all(messages.map(async (msg) => {
//       let senderName = 'Unknown User';
      
//       if (msg.senderType === 'admin') {
//         senderName = 'Admin';
//       } else if (msg.senderType === 'student') {
//         const student = await Student.findOne({ userId: msg.senderId });
//         senderName = student?.studentName || 'Student';
//       } else if (msg.senderType === 'teacher') {
//         const teacher = await Teacher.findOne({ userId: msg.senderId });
//         senderName = teacher?.name || 'Teacher';
//       }
      
//       return {
//         _id: msg._id,
//         senderId: msg.senderId,
//         sender: msg.senderType,
//         senderType: msg.senderType,
//         senderName,
//         message: msg.message,
//         text: msg.message,
//         timestamp: msg.timestamp,
//         readBy: msg.readBy
//       };
//     }));

//     res.json({ 
//       messages: transformedMessages,
//       totalMessages,
//       hasMore: startIndex > 0
//     });
//   } catch (error) {
//     console.error('Get messages error:', error);
//     res.status(500).json({ message: 'Server error' });
//   }
// });
router.get('/my-chats', auth, async (req, res) => {
  try {
    const userId = req.user._id;

    const chats = await Chat.find({
      'participants.userId': userId,
      isActive: true
    })
      .populate('participants.userId', 'username')
      .sort({ lastActivity: -1 });

    const chatsWithDetails = chats.map(chat => {
      const me = chat.participants.find(p => p.userId.toString() === userId.toString());
      const other = chat.participants.find(p => p.userId.toString() !== userId.toString());

      return {
        _id: chat._id,
        lastActivity: chat.lastActivity,
        otherParticipant: {
          id: other.userId._id,
          name: other.userId.username,
          type: other.userType
        },
        unreadCount: me?.unreadCount || 0,
        lastMessage: chat.messages.length ? chat.messages[chat.messages.length - 1] : null
      };
    });

    res.json({ chats: chatsWithDetails });
  } catch (err) {
    console.error('❌ Get chats error:', err);
    res.status(500).json({ message: 'Server error' });
  }
});

router.get('/:chatId/messages', auth, async (req, res) => {
  try {
    const userId = req.user._id;
    const { page = 1, limit = 50 } = req.query;

    const chat = await Chat.findById(req.params.chatId);
    if (!chat) return res.status(404).json({ message: 'Chat not found' });

    const me = chat.participants.find(p => p.userId.toString() === userId.toString());
    if (!me) return res.status(403).json({ message: 'Access denied' });

    // Reset unread count
    me.unreadCount = 0;
    await chat.save();

    // Emit reset
    req.app.get('io').to(`user-${userId}`).emit('chatUnreadUpdate', {
      chatId: chat._id,
      unreadCount: 0
    });

    // Pagination
    const totalMessages = chat.messages.length;
    const startIndex = Math.max(0, totalMessages - page * limit);
    const endIndex = totalMessages - (page - 1) * limit;
    const messages = chat.messages.slice(startIndex, endIndex);

    res.json({
      messages,
      totalMessages,
      hasMore: startIndex > 0
    });
  } catch (err) {
    console.error('❌ Get messages error:', err);
    res.status(500).json({ message: 'Server error' });
  }
});



router.get('/admin/all-chats', adminAuth, async (req, res) => {
  try {
    const { page = 1, limit = 20 } = req.query;

    const chats = await Chat.find({ isActive: true })
      .populate('participants.userId', 'username')
      .sort({ lastActivity: -1 })
      .limit(limit * 1)
      .skip((page - 1) * limit);

    const chatsWithDetails = await Promise.all(
      chats.map(async (chat) => {
        const participantDetails = await Promise.all(
          chat.participants.map(async (participant) => {
            let details = { username: participant.userId?.username || 'Unknown' };
            
            if (participant.userId) {
              if (participant.userType === 'student') {
                const student = await Student.findOne({ userId: participant.userId._id });
                details.name = student?.studentName;
              } else if (participant.userType === 'teacher') {
                const teacher = await Teacher.findOne({ userId: participant.userId._id });
                details.name = teacher?.name;
              } else {
                details.name = 'Admin';
              }
            } else {
              details.name = 'Unknown User';
            }
            
            return { ...details, type: participant.userType };
          })
        );

        return {
          ...chat.toObject(),
          participantDetails,
          messageCount: chat.messages.length,
          lastMessage: chat.messages[chat.messages.length - 1]
        };
      })
    );

    const total = await Chat.countDocuments({ isActive: true });

    res.json({
      chats: chatsWithDetails,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total
    });
  } catch (error) {
    console.error('Get all chats error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Mark specific messages as read
router.put('/:chatId/messages/read', auth, async (req, res) => {
  try {
    const userId = req.user._id;
    const { messageIds } = req.body; // Array of message IDs to mark as read

    const chat = await Chat.findById(req.params.chatId);
    if (!chat) {
      return res.status(404).json({ message: 'Chat not found' });
    }

    const isParticipant = chat.participants.some(p => p.userId.toString() === userId.toString());
    if (!isParticipant) {
      return res.status(403).json({ message: 'Access denied' });
    }

    let hasUpdates = false;
    
    if (messageIds && Array.isArray(messageIds)) {
      // Mark specific messages as read
      chat.messages.forEach(msg => {
        if (messageIds.includes(msg._id.toString()) && 
            msg.senderId.toString() !== userId.toString()) {
          const alreadyRead = msg.readBy.some(r => r.userId.toString() === userId.toString());
          if (!alreadyRead) {
            msg.readBy.push({ userId: userId, readAt: new Date() });
            hasUpdates = true;
          }
        }
      });
    } else {
      // Mark all unread messages as read
      chat.messages.forEach(msg => {
        if (msg.senderId.toString() !== userId.toString()) {
          const alreadyRead = msg.readBy.some(r => r.userId.toString() === userId.toString());
          if (!alreadyRead) {
            msg.readBy.push({ userId: userId, readAt: new Date() });
            hasUpdates = true;
          }
        }
      });
    }

    if (hasUpdates) {
      await chat.save();
    }

    res.json({ message: 'Messages marked as read successfully' });
  } catch (error) {
    console.error('Mark messages as read error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Get read status for messages
router.get('/:chatId/read-status', auth, async (req, res) => {
  try {
    const userId = req.user._id;

    const chat = await Chat.findById(req.params.chatId)
      .populate('messages.readBy.userId', 'username');
      
    if (!chat) {
      return res.status(404).json({ message: 'Chat not found' });
    }

    const isParticipant = chat.participants.some(p => p.userId.toString() === userId.toString());
    if (!isParticipant) {
      return res.status(403).json({ message: 'Access denied' });
    }

    const messagesWithReadStatus = chat.messages.map(msg => ({
      messageId: msg._id,
      senderId: msg.senderId,
      message: msg.message,
      timestamp: msg.timestamp,
      readBy: msg.readBy.filter(read => read.userId).map(read => ({
        userId: read.userId._id,
        username: read.userId.username,
        readAt: read.readAt
      })),
      isReadByCurrentUser: msg.readBy.some(r => r.userId && r.userId._id.toString() === userId.toString()),
      readCount: msg.readBy.length
    }));

    res.json({ messages: messagesWithReadStatus });
  } catch (error) {
    console.error('Get read status error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

router.delete('/:chatId', adminAuth, async (req, res) => {
  try {
    const chat = await Chat.findByIdAndUpdate(
      req.params.chatId,
      { isActive: false },
      { new: true }
    );

    if (!chat) {
      return res.status(404).json({ message: 'Chat not found' });
    }

    res.json({ message: 'Chat deactivated successfully' });
  } catch (error) {
    console.error('Delete chat error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

export default router;