import express from 'express';
import Notification from '../../models/Notification.js';
import { auth } from '../../middleware/advancedAuth.js';

const router = express.Router();

// Get all notifications for a user
router.get('/', auth, async (req, res) => {
  try {
    const { userType } = req.user;
    const userId = req.user._id;
    const { limit = 100, skip = 0 } = req.query;

    console.log('usertype and userId>>',userType,userId)

    const notifications = await Notification.find({
      userId,
      userType
    })
      .sort({ createdAt: -1 })
      .skip(parseInt(skip))
      .limit(parseInt(limit));

    const unreadCount = await Notification.countDocuments({
      userId,
      userType,
      isRead: false
    });

    const totalCount = await Notification.countDocuments({
      userId,
      userType
    });

    res.json({
      notifications,
      unreadCount,
      totalCount,
      hasMore: (parseInt(skip) + notifications.length) < totalCount
    });
  } catch (error) {
    console.error('Error fetching notifications:', error);
    res.status(500).json({ error: error.message });
  }
});

// Get unread count
router.get('/unread-count', auth, async (req, res) => {
  try {
    const { userType } = req.user;
    const userId = req.user._id;

    const count = await Notification.countDocuments({
      userId,
      userType,
      isRead: false
    });

    res.json({ count });
  } catch (error) {
    console.error('Error fetching unread count:', error);
    res.status(500).json({ error: error.message });
  }
});

// Mark notification as read
router.patch('/:id/read', auth, async (req, res) => {
  try {
    const { id } = req.params;
    const userId = req.user._id;

    const notification = await Notification.findOneAndUpdate(
      { _id: id, userId },
      { isRead: true },
      { new: true }
    );

    if (!notification) {
      return res.status(404).json({ error: 'Notification not found' });
    }

    res.json({ notification });
  } catch (error) {
    console.error('Error marking notification as read:', error);
    res.status(500).json({ error: error.message });
  }
});

// Mark all notifications as read
router.patch('/mark-all-read', auth, async (req, res) => {
  try {
    const { userType } = req.user;
    const userId = req.user._id;

    await Notification.updateMany(
      { userId, userType, isRead: false },
      { isRead: true }
    );

    res.json({ message: 'All notifications marked as read' });
  } catch (error) {
    console.error('Error marking all notifications as read:', error);
    res.status(500).json({ error: error.message });
  }
});

// Delete a notification
router.delete('/:id', auth, async (req, res) => {
  try {
    const { id } = req.params;
    const userId = req.user._id;

    const notification = await Notification.findOneAndDelete({
      _id: id,
      userId
    });

    if (!notification) {
      return res.status(404).json({ error: 'Notification not found' });
    }

    res.json({ message: 'Notification deleted' });
  } catch (error) {
    console.error('Error deleting notification:', error);
    res.status(500).json({ error: error.message });
  }
});

// Create a notification (internal use or for testing)
router.post('/create', auth, async (req, res) => {
  try {
    const { userId, userType, type, title, message, relatedId, relatedModel, priority, actionUrl } = req.body;

    const notification = new Notification({
      userId,
      userType,
      type,
      title,
      message,
      relatedId,
      relatedModel,
      priority: priority || 'medium',
      actionUrl
    });

    await notification.save();

    // Emit socket event (will be handled by socket.io)
    const io = req.app.get('io');
    if (io) {
      io.to(`user-${userId}`).emit('new_notification', notification);
    }

    res.status(201).json({ notification });
  } catch (error) {
    console.error('Error creating notification:', error);
    res.status(500).json({ error: error.message });
  }
});

export default router;
