import express from 'express';
import multer from 'multer';
import path from 'path';
import fs from 'fs';
import Assignment from '../../models/teacher/Assignment.js';
import AssignmentSubmission from '../../models/student/AssignmentSubmission.js';
import { studentAuth } from '../../middleware/auth.js';

const router = express.Router();

// Configure multer for student file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = path.join(__dirname, '../../uploads/assignments/student');
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, 'submission-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({
  storage: storage,
  limits: {
    fileSize: 10 * 1024 * 1024, // 10MB limit
  },
  fileFilter: (req, file, cb) => {
    const allowedTypes = [
      'application/pdf',
      'application/msword',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
      'application/vnd.ms-powerpoint',
      'application/vnd.openxmlformats-officedocument.presentationml.presentation',
      'text/plain',
      'image/jpeg',
      'image/png',
      'image/gif'
    ];
    
    if (allowedTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error('Invalid file type. Only PDF, Word, PowerPoint, text, and image files are allowed.'));
    }
  }
});

// Get assignments for a student
router.get('/', studentAuth, async (req, res) => {
  try {
    const { studentId, status, page = 1, limit = 10 } = req.query;
    
    if (!studentId) {
      return res.status(400).json({ success: false, error: 'Student ID is required' });
    }
    
    let filter = {
      'assignedStudents.studentId': studentId,
      status: 'published'
    };
    
    const skip = (page - 1) * limit;
    
    const assignments = await Assignment.find(filter)
      .sort({ dueDate: 1 })
      .skip(skip)
      .limit(parseInt(limit));
    
    // Get submission status for each assignment
    const assignmentsWithSubmissions = await Promise.all(
      assignments.map(async (assignment) => {
        const submission = await AssignmentSubmission.findOne({
          assignmentId: assignment._id,
          studentId: studentId
        });
        
        return {
          ...assignment.toObject(),
          submissionStatus: submission ? submission.status : 'not_submitted',
          submission: submission || null,
          isOverdue: assignment.isOverdue(),
          finalGrade: submission ? await submission.getFinalGrade() : null
        };
      })
    );
    
    // Filter by submission status if requested
    let filteredAssignments = assignmentsWithSubmissions;
    if (status) {
      if (status === 'pending') {
        filteredAssignments = assignmentsWithSubmissions.filter(a => a.submissionStatus === 'not_submitted');
      } else if (status === 'submitted') {
        filteredAssignments = assignmentsWithSubmissions.filter(a => ['submitted', 'under_review', 'graded', 'returned'].includes(a.submissionStatus));
      } else if (status === 'overdue') {
        filteredAssignments = assignmentsWithSubmissions.filter(a => a.isOverdue && a.submissionStatus === 'not_submitted');
      } else {
        filteredAssignments = assignmentsWithSubmissions.filter(a => a.submissionStatus === status);
      }
    }
    
    const total = await Assignment.countDocuments(filter);
    
    res.json({
      success: true,
      assignments: filteredAssignments,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get single assignment details for student
router.get('/:id', async (req, res) => {
  try {
    const { studentId } = req.query;
    
    if (!studentId) {
      return res.status(400).json({ success: false, error: 'Student ID is required' });
    }
    
    const assignment = await Assignment.findById(req.params.id);
    if (!assignment) {
      return res.status(404).json({ success: false, error: 'Assignment not found' });
    }
    
    // Check if student is assigned to this assignment
    const isAssigned = assignment.assignedStudents.some(student => student.studentId === studentId);
    if (!isAssigned) {
      return res.status(403).json({ success: false, error: 'You are not assigned to this assignment' });
    }
    
    // Get student's submission if exists
    const submission = await AssignmentSubmission.findOne({
      assignmentId: assignment._id,
      studentId: studentId
    });
    
    res.json({
      success: true,
      assignment: {
        ...assignment.toObject(),
        submissionStatus: submission ? submission.status : 'not_submitted',
        submission: submission || null,
        isOverdue: assignment.isOverdue(),
        finalGrade: submission ? await submission.getFinalGrade() : null
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Submit assignment
router.post('/:id/submit', upload.array('attachments', 5), async (req, res) => {
  try {
    const { studentId, studentName, submissionText } = req.body;
    const assignmentId = req.params.id;
    
    if (!studentId || !studentName) {
      return res.status(400).json({ success: false, error: 'Student ID and name are required' });
    }
    
    const assignment = await Assignment.findById(assignmentId);
    if (!assignment) {
      return res.status(404).json({ success: false, error: 'Assignment not found' });
    }
    
    if (assignment.status !== 'published') {
      return res.status(400).json({ success: false, error: 'Assignment is not available for submission' });
    }
    
    // Check if student is assigned to this assignment
    const isAssigned = assignment.assignedStudents.some(student => student.studentId === studentId);
    if (!isAssigned) {
      return res.status(403).json({ success: false, error: 'You are not assigned to this assignment' });
    }
    
    // Check if already submitted
    const existingSubmission = await AssignmentSubmission.findOne({ assignmentId, studentId });
    if (existingSubmission && !existingSubmission.revisionRequested) {
      return res.status(400).json({ success: false, error: 'Assignment already submitted' });
    }
    
    // Check if late submission is allowed
    const isLate = assignment.isOverdue();
    if (isLate && !assignment.allowLateSubmission) {
      return res.status(400).json({ success: false, error: 'Late submissions are not allowed for this assignment' });
    }
    
    const attachments = req.files ? req.files.map(file => ({
      filename: file.filename,
      originalName: file.originalname,
      path: file.path,
      size: file.size,
      mimetype: file.mimetype
    })) : [];
    
    let submission;
    
    if (existingSubmission && existingSubmission.revisionRequested) {
      // Handle revision submission
      await existingSubmission.createNewVersion({
        submissionText,
        attachments
      });
      submission = existingSubmission;
    } else {
      // Create new submission
      submission = new AssignmentSubmission({
        assignmentId,
        studentId,
        studentName,
        submissionText,
        attachments,
        isLateSubmission: isLate,
        submittedAt: new Date()
      });
      
      await submission.save();
    }
    
    // Emit socket event for real-time notifications
    req.app.get('io').emit('assignmentSubmitted', {
      assignmentId,
      submissionId: submission._id,
      studentId,
      studentName,
      isRevision: existingSubmission && existingSubmission.revisionRequested
    });
    
    res.status(201).json({ success: true, submission });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Update submission (before final submission)
router.put('/:id/submission', upload.array('attachments', 5), async (req, res) => {
  try {
    const { studentId, submissionText } = req.body;
    const assignmentId = req.params.id;
    
    const submission = await AssignmentSubmission.findOne({ assignmentId, studentId });
    if (!submission) {
      return res.status(404).json({ success: false, error: 'Submission not found' });
    }
    
    if (submission.status !== 'submitted') {
      return res.status(400).json({ success: false, error: 'Cannot update submission that has been reviewed' });
    }
    
    const assignment = await Assignment.findById(assignmentId);
    if (assignment.isOverdue() && !assignment.allowLateSubmission) {
      return res.status(400).json({ success: false, error: 'Assignment deadline has passed' });
    }
    
    // Update submission text
    if (submissionText !== undefined) {
      submission.submissionText = submissionText;
    }
    
    // Handle new file uploads
    if (req.files && req.files.length > 0) {
      const newAttachments = req.files.map(file => ({
        filename: file.filename,
        originalName: file.originalname,
        path: file.path,
        size: file.size,
        mimetype: file.mimetype
      }));
      submission.attachments.push(...newAttachments);
    }
    
    submission.submittedAt = new Date();
    await submission.save();
    
    res.json({ success: true, submission });
  } catch (error) {
    res.status(400).json({ success: false, error: error.message });
  }
});

// Get student's submission history
router.get('/submissions/history/:studentId', async (req, res) => {
  try {
    const { status, page = 1, limit = 10 } = req.query;
    const filter = { studentId: req.params.studentId };
    
    if (status) filter.status = status;
    
    const skip = (page - 1) * limit;
    
    const submissions = await AssignmentSubmission.find(filter)
      .populate({
        path: 'assignmentId',
        model: 'Assignment',
        select: 'title subject dueDate maxMarks'
      })
      .sort({ submittedAt: -1 })
      .skip(skip)
      .limit(parseInt(limit));
    
    // Calculate final grades
    const submissionsWithGrades = await Promise.all(
      submissions.map(async (submission) => ({
        ...submission.toObject(),
        finalGrade: await submission.getFinalGrade()
      }))
    );
    
    const total = await AssignmentSubmission.countDocuments(filter);
    
    res.json({
      success: true,
      submissions: submissionsWithGrades,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        pages: Math.ceil(total / limit)
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Get specific submission details
router.get('/submissions/:submissionId', async (req, res) => {
  try {
    const submission = await AssignmentSubmission.findById(req.params.submissionId)
      .populate({
        path: 'assignmentId',
        model: 'Assignment',
        select: 'title subject dueDate maxMarks instructions attachments'
      });
    
    if (!submission) {
      return res.status(404).json({ success: false, error: 'Submission not found' });
    }
    
    const finalGrade = await submission.getFinalGrade();
    
    res.json({
      success: true,
      submission: {
        ...submission.toObject(),
        finalGrade
      }
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Download assignment attachment (for students)
router.get('/:id/attachments/:filename', async (req, res) => {
  try {
    const { studentId } = req.query;
    
    const assignment = await Assignment.findById(req.params.id);
    if (!assignment) {
      return res.status(404).json({ success: false, error: 'Assignment not found' });
    }
    
    // Check if student is assigned to this assignment
    const isAssigned = assignment.assignedStudents.some(student => student.studentId === studentId);
    if (!isAssigned) {
      return res.status(403).json({ success: false, error: 'Access denied' });
    }
    
    const attachment = assignment.attachments.find(att => att.filename === req.params.filename);
    if (!attachment) {
      return res.status(404).json({ success: false, error: 'Attachment not found' });
    }
    
    const filePath = attachment.path;
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({ success: false, error: 'File not found on server' });
    }
    
    res.download(filePath, attachment.originalName);
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Download submission attachment
router.get('/submissions/:submissionId/attachments/:filename', async (req, res) => {
  try {
    const submission = await AssignmentSubmission.findById(req.params.submissionId);
    if (!submission) {
      return res.status(404).json({ success: false, error: 'Submission not found' });
    }
    
    const attachment = submission.attachments.find(att => att.filename === req.params.filename);
    if (!attachment) {
      return res.status(404).json({ success: false, error: 'Attachment not found' });
    }
    
    const filePath = attachment.path;
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({ success: false, error: 'File not found on server' });
    }
    
    res.download(filePath, attachment.originalName);
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Remove attachment from submission (before grading)
router.delete('/submissions/:submissionId/attachments/:filename', async (req, res) => {
  try {
    const submission = await AssignmentSubmission.findById(req.params.submissionId);
    if (!submission) {
      return res.status(404).json({ success: false, error: 'Submission not found' });
    }
    
    if (submission.status !== 'submitted') {
      return res.status(400).json({ success: false, error: 'Cannot modify submission that has been reviewed' });
    }
    
    const attachmentIndex = submission.attachments.findIndex(att => att.filename === req.params.filename);
    if (attachmentIndex === -1) {
      return res.status(404).json({ success: false, error: 'Attachment not found' });
    }
    
    const attachment = submission.attachments[attachmentIndex];
    
    // Delete file from filesystem
    if (fs.existsSync(attachment.path)) {
      fs.unlinkSync(attachment.path);
    }
    
    // Remove from database
    submission.attachments.splice(attachmentIndex, 1);
    await submission.save();
    
    res.json({ success: true, message: 'Attachment removed successfully' });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

export default router;