import express from 'express';
import mongoose from 'mongoose';
import crypto from 'crypto';
import Teacher from '../../models/teacher/Teacher.js';
import Student from '../../models/student/Student.js';
import Class from '../../models/Class.js';

const router = express.Router();

// Helper function to convert development IDs to valid ObjectIds
const getValidObjectId = (id) => {
  if (mongoose.Types.ObjectId.isValid(id)) {
    return id;
  } else if (typeof id === 'string' && id.startsWith('dev-')) {
    const hash = crypto.createHash('md5').update(id).digest('hex');
    const objectIdHex = hash.substring(0, 24);
    return new mongoose.Types.ObjectId(objectIdHex);
  }
  return id;
};

// Get students assigned to teacher
router.get('/students', async (req, res) => {
  try {
    const teacherId = getValidObjectId(req.user._id);

    // Fetch students from both sources: Teacher schema (assigned students) and Class schema (scheduled classes)

    // 1. Get assigned students from Teacher schema
    const teacher = await Teacher.findById(teacherId)
      .populate({
        path: 'assignedStudents.studentId',
        select: 'studentName email class subjects phoneNumber country profilePicture'
      });

    // 2. Get students from Class schema (students with scheduled classes)
    const classStudents = await Class.find({ teacherId,      status: { $in: ['scheduled', 'ongoing', 'completed'] }
 })
      .populate({
        path: 'studentId',
        select: 'studentName email class subjects phoneNumber country profilePicture'
      })
      .select('studentId subject scheduledDate status')
      .sort({ scheduledDate: -1 });

    let students = [];
    const studentMap = new Map(); // To avoid duplicates and merge data

    // Process assigned students from Teacher schema
    if (teacher && teacher.assignedStudents && teacher.assignedStudents.length > 0) {
      teacher.assignedStudents.forEach(assignment => {
        if (assignment.studentId) {
          const studentKey = assignment.studentId._id.toString();
          studentMap.set(studentKey, {
            student: {
              _id: assignment.studentId._id,
              studentName: assignment.studentId.studentName,
              email: assignment.studentId.email,
              class: assignment.studentId.class || 'No Class',
              subjects: assignment.studentId.subjects || [assignment.subject],
              phoneNumber: assignment.studentId.phoneNumber,
              country: assignment.studentId.country || 'Unknown',
              profilePicture: assignment.studentId.profilePicture
            },
            subject: assignment.subject,
            canChat: true,
            totalClasses: 0,
            completedClasses: 0,
            attendanceRate: 0,
            lastClass: null,
            source: 'teacher_assignment'
          });
        }
      });
    }

    // Process students from Class schema and calculate stats
    if (classStudents && classStudents.length > 0) {
      classStudents.forEach(classItem => {
        if (classItem.studentId) {
          const studentKey = classItem.studentId._id.toString();

          if (studentMap.has(studentKey)) {
            // Update existing student with class information
            const existingStudent = studentMap.get(studentKey);
            existingStudent.totalClasses = (existingStudent.totalClasses || 0) + 1;

            // Count completed classes for attendance
            if (classItem.status === 'completed') {
              existingStudent.completedClasses = (existingStudent.completedClasses || 0) + 1;
            }

            // Update last class date
            const classDate = classItem.scheduledDate ? classItem.scheduledDate.toISOString().split('T')[0] : null;
            if (!existingStudent.lastClass || (classDate && classDate > existingStudent.lastClass)) {
              existingStudent.lastClass = classDate;
            }

            existingStudent.source = 'both'; // Mark as found in both sources
          } else {
            // Add new student from class schedule
            studentMap.set(studentKey, {
              student: {
                _id: classItem.studentId._id,
                studentName: classItem.studentId.studentName,
                email: classItem.studentId.email,
                class: classItem.studentId.class || 'No Class',
                subjects: classItem.studentId.subjects || [classItem.subject],
                phoneNumber: classItem.studentId.phoneNumber,
                country: classItem.studentId.country || 'Unknown',
                profilePicture: classItem.studentId.profilePicture
              },
              subject: classItem.subject,
              canChat: true,
              totalClasses: 1,
              completedClasses: classItem.status === 'completed' ? 1 : 0,
              attendanceRate: 0,
              lastClass: classItem.scheduledDate ? classItem.scheduledDate.toISOString().split('T')[0] : null,
              source: 'class_schedule'
            });
          }
        }
      });
    }

    // Convert Map to array and calculate attendance rates
    students = Array.from(studentMap.values()).map(student => {
      // Calculate attendance rate
      if (student.totalClasses > 0) {
        student.attendanceRate = Math.round((student.completedClasses / student.totalClasses) * 100);
      }
      return student;
    });

    // Fetch last message for each student
    const Message = mongoose.model('Message');
    const studentsWithMessages = await Promise.all(
      students.map(async (studentData) => {
        const studentId = studentData.student._id;

        // Find last message between teacher and student
        const lastMessage = await Message.findOne({
          $or: [
            { senderId: teacherId, receiverId: studentId, isGroupMessage: false },
            { senderId: studentId, receiverId: teacherId, isGroupMessage: false }
          ]
        })
        .sort({ timestamp: -1 })
        .select('content timestamp')
        .lean();

        return {
          ...studentData,
          lastMessage: lastMessage?.content || null,
          lastMessageTime: lastMessage?.timestamp || null
        };
      })
    );

    res.json({
      success: true,
      data: studentsWithMessages,
      meta: {
        totalStudents: students.length,
        sources: {
          teacherAssignments: students.filter(s => s.source === 'teacher_assignment' || s.source === 'both').length,
          classSchedules: students.filter(s => s.source === 'class_schedule' || s.source === 'both').length,
          both: students.filter(s => s.source === 'both').length
        }
      }
    });

  } catch (error) {
    console.error('Get students error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch students',
      error: error.message
    });
  }
});

// Simple students endpoint for homework (returns simplified format)
router.get('/students/simple', async (req, res) => {
  try {
    const teacherId = getValidObjectId(req.user._id);

    // Fetch students from both sources: Teacher schema (assigned students) and Class schema (scheduled classes)

    // 1. Get assigned students from Teacher schema
    const teacher = await Teacher.findById(teacherId)
      .populate({
        path: 'assignedStudents.studentId',
        select: 'studentName email class subjects phoneNumber'
      });

    // 2. Get students from Class schema (students with scheduled classes)
    const classStudents = await Class.find({
      teacherId,
      status: { $in: ['scheduled', 'ongoing', 'completed'] }
    })
      .populate({
        path: 'studentId',
        select: 'studentName email class subjects phoneNumber'
      })
      .select('studentId subject scheduledDate status')
      .sort({ scheduledDate: -1 });

    let students = [];
    const studentMap = new Map(); // To avoid duplicates and merge data

    // Process assigned students from Teacher schema
    if (teacher && teacher.assignedStudents && teacher.assignedStudents.length > 0) {
      teacher.assignedStudents.forEach(assignment => {
        if (assignment.studentId) {
          const studentKey = assignment.studentId._id.toString();
          studentMap.set(studentKey, {
            _id: assignment.studentId._id,
            studentName: assignment.studentId.studentName,
            name: assignment.studentId.studentName, // For homework component compatibility
            email: assignment.studentId.email,
            class: assignment.studentId.class || 'No Class',
            subjects: assignment.studentId.subjects || [assignment.subject],
            phoneNumber: assignment.studentId.phoneNumber,
            canChat: true
          });
        }
      });
    }

    // Process students from Class schema
    if (classStudents && classStudents.length > 0) {
      classStudents.forEach(classItem => {
        if (classItem.studentId) {
          const studentKey = classItem.studentId._id.toString();

          if (!studentMap.has(studentKey)) {
            // Add new student from class schedule
            studentMap.set(studentKey, {
              _id: classItem.studentId._id,
              studentName: classItem.studentId.studentName,
              name: classItem.studentId.studentName, // For homework component compatibility
              email: classItem.studentId.email,
              class: classItem.studentId.class || 'No Class',
              subjects: classItem.studentId.subjects || [classItem.subject],
              phoneNumber: classItem.studentId.phoneNumber,
              canChat: true
            });
          }
        }
      });
    }

    // Convert Map to array
    students = Array.from(studentMap.values());


    res.json({
      success: true,
      data: students
    });

  } catch (error) {
    console.error('Get simple students error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch students',
      error: error.message
    });
  }
});

router.get('/dashboard', async (req, res) => {
  try {
    const teacherId = getValidObjectId(req.user._id);

    // Get teacher info
    const teacher = await Teacher.findById(teacherId);
    if (!teacher) {
      return res.status(404).json({ message: 'Teacher not found' });
    }

    // Import Homework and QuickTest models
    const Homework = mongoose.model('Homework');
    const QuickTest = mongoose.model('QuickTest');

    // Get today's date range
    const startOfToday = new Date();
    startOfToday.setHours(0, 0, 0, 0);
    const endOfToday = new Date();
    endOfToday.setHours(23, 59, 59, 999);

    // Get tomorrow's start for upcoming classes
    const startOfTomorrow = new Date(startOfToday);
    startOfTomorrow.setDate(startOfTomorrow.getDate() + 1);

    // Get classes for today
    const todaysClasses = await Class.find({
      teacherId: teacherId,
      scheduledDate: {
        $gte: startOfToday,
        $lte: endOfToday
      },
      isDeleted: false
    })
    .populate('studentId', 'studentName email class')
    .sort({ scheduledDate: 1 });

    // Get upcoming classes (next 7 days excluding today)
    const next7Days = new Date(startOfTomorrow);
    next7Days.setDate(next7Days.getDate() + 7);

    const upcomingClasses = await Class.find({
      teacherId: teacherId,
      scheduledDate: {
        $gte: startOfTomorrow,
        $lt: next7Days
      },
      status: { $in: ['scheduled'] },
      isDeleted: false
    })
    .populate('studentId', 'studentName email class')
    .sort({ scheduledDate: 1 })
    .limit(10);

    // Get current month stats
    const startOfMonth = new Date();
    startOfMonth.setDate(1);
    startOfMonth.setHours(0, 0, 0, 0);

    const endOfMonth = new Date();
    endOfMonth.setMonth(endOfMonth.getMonth() + 1);
    endOfMonth.setDate(0);
    endOfMonth.setHours(23, 59, 59, 999);

    const monthlyClasses = await Class.find({
      teacherId: teacherId,
      scheduledDate: {
        $gte: startOfMonth,
        $lte: endOfMonth
      },
      isDeleted: false
    });

    // Calculate monthly stats
    const completedClasses = monthlyClasses.filter(c => c.status === 'completed');
    // Calculate total monthly hours (all classes regardless of status)
    const totalMonthlyHours = monthlyClasses.reduce((sum, c) => sum + (c.duration || 60), 0) / 60;

    // Get unique students for this month
    const monthlyStudentIds = new Set(monthlyClasses.map(c => c.studentId?.toString()).filter(Boolean));

    // Calculate weekly hours (last 7 days)
    const last7Days = new Date();
    last7Days.setDate(last7Days.getDate() - 7);

    const weeklyClasses = await Class.find({
      teacherId: teacherId,
      scheduledDate: {
        $gte: last7Days,
        $lte: new Date()
      },
      status: 'completed',
      isDeleted: false
    });

    const weeklyHours = weeklyClasses.reduce((sum, c) => sum + (c.duration || 60), 0) / 60;

    // Get total unique students
    const allStudents = await Class.distinct('studentId', { teacherId: teacherId, isDeleted: false });
    const totalStudents = allStudents.length;

    // Get total completed classes (all time)
    const allCompletedClasses = await Class.countDocuments({
      teacherId: teacherId,
      status: 'completed',
      isDeleted: false
    });

    // Calculate total completed hours (all time)
    const allCompletedClassesFull = await Class.find({
      teacherId: teacherId,
      status: 'completed',
      isDeleted: false
    });
    const totalCompletedHours = allCompletedClassesFull.reduce((sum, c) => sum + (c.duration || 60), 0) / 60;

    // Calculate assignments (homework + quick tests created by teacher)
    const totalHomework = await Homework.countDocuments({ teacherId: teacherId });
    const totalQuickTests = await QuickTest.countDocuments({ teacherId: teacherId });
    const totalAssignments = totalHomework + totalQuickTests;

    // Get submitted homework and quick tests
    const submittedHomework = await Homework.countDocuments({
      teacherId: teacherId,
      'submission.submittedAt': { $exists: true }
    });
    const submittedQuickTests = await QuickTest.countDocuments({
      teacherId: teacherId,
      'submission.submittedAt': { $exists: true }
    });
    const completedAssignments = submittedHomework + submittedQuickTests;

    res.json({
      teacher: {
        name: teacher.name,
        email: teacher.email,
        subjects: teacher.subjects || [],
        assignedStudentsCount: totalStudents,
        salary: teacher.salary,
        salaryType: teacher.salaryType
      },
      todaysClasses: todaysClasses.map(cls => ({
        _id: cls._id,
        subject: cls.subject,
        scheduledDate: cls.scheduledDate,
        startTime: cls.startTime,
        endTime: cls.endTime,
        duration: cls.duration,
        studentId: cls.studentId,
        status: cls.status,
        googleMeetLink: cls.googleMeetLink
      })),
      upcomingClasses: upcomingClasses.map(cls => ({
        _id: cls._id,
        subject: cls.subject,
        scheduledDate: cls.scheduledDate,
        startTime: cls.startTime,
        endTime: cls.endTime,
        duration: cls.duration,
        studentId: cls.studentId,
        status: cls.status
      })),
      stats: {
        totalStudents: totalStudents,
        monthlyClasses: monthlyClasses.length,
        completedClasses: completedClasses.length,
        allCompletedClasses: allCompletedClasses, // Total completed classes (all time)
        totalCompletedHours: Math.round(totalCompletedHours * 10) / 10, // Total completed hours (all time)
        todayClasses: todaysClasses.length,
        completedToday: todaysClasses.filter(c => c.status === 'completed').length,
        monthlyHours: Math.round(totalMonthlyHours * 10) / 10,
        weeklyHours: Math.round(weeklyHours * 10) / 10,
        monthlyEarnings: teacher.salaryType === 'hourly'
          ? Math.round(totalMonthlyHours * (teacher.salary?.amount || 0))
          : teacher.salary?.amount || 0,
        assignmentsTotal: totalAssignments, // Total homework + quick tests created
        assignmentsCompleted: completedAssignments, // Submitted homework + quick tests
        assignmentsPending: totalAssignments - completedAssignments,
        studentsActive: monthlyStudentIds.size
      }
    });
  } catch (error) {
    console.error('Dashboard error:', error);
    res.status(500).json({ message: 'Server error', error: error.message });
  }
});

export default router;