import Notification from '../models/Notification.js';
import mongoose from 'mongoose';

/**
 * Create and send a notification
 * @param {Object} params - Notification parameters
 * @param {String|ObjectId} params.userId - User ID to send notification to
 * @param {String} params.userType - User type (admin/teacher/student)
 * @param {String} params.type - Notification type
 * @param {String} params.title - Notification title
 * @param {String} params.message - Notification message
 * @param {String} params.relatedId - Related document ID (optional)
 * @param {String} params.relatedModel - Related model name (optional)
 * @param {String} params.priority - Priority level (low/medium/high)
 * @param {String} params.actionUrl - Action URL (optional)
 * @param {Object} io - Socket.IO instance
 */
export const createNotification = async ({
  userId,
  userType,
  type,
  title,
  message,
  relatedId = null,
  relatedModel = null,
  priority = 'medium',
  actionUrl = null,
  io
}) => {
  try {
    // Convert userId to ObjectId if it's a string
    const userObjectId = typeof userId === 'string' ? new mongoose.Types.ObjectId(userId) : userId;

    const notification = new Notification({
      userId: userObjectId,
      userType,
      type,
      title,
      message,
      relatedId,
      relatedModel,
      priority,
      actionUrl
    });

    await notification.save();

    // Emit socket event to user
    if (io) {
      const roomName = `user-${userId}`;
      console.log(`📢 Emitting notification to room: ${roomName}`);
      console.log(`   Notification ID: ${notification._id}`);
      console.log(`   Title: ${title}`);
      console.log(`   Type: ${type}`);
      console.log(`   User Type: ${userType}`);
      console.log(`   User ID: ${userId}`);

      // Convert Mongoose document to plain JavaScript object
      const notificationObject = notification.toObject();
      console.log(`   Full notification object:`, JSON.stringify(notificationObject, null, 2));

      // Get all sockets in the room for debugging
      const socketsInRoom = await io.in(roomName).fetchSockets();
      console.log(`   Sockets in room ${roomName}:`, socketsInRoom.length);
      socketsInRoom.forEach(socket => {
        console.log(`     - Socket ID: ${socket.id}, User: ${socket.userId}`);
      });

      // Emit to room
      io.to(roomName).emit('new_notification', notificationObject);
      console.log(`✅ Notification emitted successfully to room ${roomName}`);
      console.log(`   Emitted at: ${new Date().toISOString()}`);
    } else {
      console.warn(`⚠️ Socket.IO instance not available, notification NOT emitted to user-${userId}`);
    }

    return notification;
  } catch (error) {
    console.error('Error creating notification:', error);
    throw error;
  }
};

/**
 * Create notifications for multiple users
 */
export const createBulkNotifications = async (notifications, io) => {
  try {
    const createdNotifications = await Notification.insertMany(notifications);

    // Emit socket events
    if (io) {
      createdNotifications.forEach(notification => {
        // Convert Mongoose document to plain JavaScript object
        const notificationObject = notification.toObject ? notification.toObject() : notification;
        console.log(`📢 Bulk notification emitted to user-${notification.userId}`);
        io.to(`user-${notification.userId}`).emit('new_notification', notificationObject);
      });
    }

    return createdNotifications;
  } catch (error) {
    console.error('Error creating bulk notifications:', error);
    throw error;
  }
};

// Notification templates for different events
export const NotificationTemplates = {
  // Class notifications
  classScheduled: (className, date, time) => ({
    type: 'class_scheduled',
    title: 'New Class Scheduled',
    message: `${className} has been scheduled for ${date} at ${time}`,
    priority: 'high'
  }),

  classRescheduled: (className, newDate, newTime) => ({
    type: 'class_rescheduled',
    title: 'Class Rescheduled',
    message: `${className} has been rescheduled to ${newDate} at ${newTime}`,
    priority: 'high'
  }),

  classCancelled: (className) => ({
    type: 'class_cancelled',
    title: 'Class Cancelled',
    message: `${className} has been cancelled`,
    priority: 'high'
  }),

  // QuickTest notifications
  quicktestAssigned: (testName, dueDate) => ({
    type: 'quicktest_assigned',
    title: 'New Quick Test Assigned',
    message: `${testName} has been assigned. Due date: ${dueDate}`,
    priority: 'medium'
  }),

  quicktestGraded: (testName, score) => ({
    type: 'quicktest_graded',
    title: 'Quick Test Graded',
    message: `Your ${testName} has been graded. Score: ${score}`,
    priority: 'medium'
  }),

  // Homework notifications
  homeworkAssigned: (subject, dueDate) => ({
    type: 'homework_assigned',
    title: 'New Homework Assigned',
    message: `New ${subject} homework assigned. Due date: ${dueDate}`,
    priority: 'medium'
  }),

  homeworkGraded: (subject, grade) => ({
    type: 'homework_graded',
    title: 'Homework Graded',
    message: `Your ${subject} homework has been graded. Grade: ${grade}`,
    priority: 'medium'
  }),

  // Invoice notifications
  invoiceGenerated: (amount, dueDate) => ({
    type: 'invoice_generated',
    title: 'New Invoice Generated',
    message: `A new invoice of ${amount} has been generated. Due: ${dueDate}`,
    priority: 'high'
  }),

  invoiceDue: (amount) => ({
    type: 'invoice_due',
    title: 'Invoice Due',
    message: `Your invoice of ${amount} is due for payment`,
    priority: 'high'
  }),

  paymentReceived: (amount) => ({
    type: 'payment_received',
    title: 'Payment Received',
    message: `Payment of £${amount} has been received`,
    priority: 'low'
  }),

  // Message notifications
  messageReceived: (senderName) => ({
    type: 'message_received',
    title: 'New Message',
    message: `You have a new message from ${senderName}`,
    priority: 'medium'
  }),

  // Trial request notifications
  trialRequest: (studentName, subject) => ({
    type: 'trial_request',
    title: 'New Trial Request',
    message: `${studentName} has requested a trial class for ${subject}`,
    priority: 'high'
  }),

  // Reschedule request notifications
  rescheduleRequest: (className, requestedDate) => ({
    type: 'reschedule_request',
    title: 'Reschedule Request',
    message: `Reschedule request for ${className} to ${requestedDate}`,
    priority: 'high'
  }),

  // Report notifications
  reportAvailable: (reportType) => ({
    type: 'report_available',
    title: 'New Report Available',
    message: `Your ${reportType} report is now available`,
    priority: 'medium'
  })
};
