#!/bin/bash

echo "🎓 EduMetrix UK Platform - Quick Development Start"
echo "================================================="
echo ""

# Kill any existing processes
echo "🧹 Cleaning up any existing processes..."
pkill -f "PORT=3001" 2>/dev/null || true
pkill -f "PORT=3002" 2>/dev/null || true
pkill -f "PORT=3003" 2>/dev/null || true
sleep 2

# Function to start a portal
start_portal() {
    local port=$1
    local type=$2
    local color=$3
    local name=$4
    
    echo "$color Starting $name Portal on port $port..."
    cd client
    nohup bash -c "PORT=$port REACT_APP_PORTAL_TYPE=$type npm start" > "../$type-portal.log" 2>&1 &
    cd ..
    echo "✅ $name Portal started"
    sleep 3
}

echo ""
echo "🚀 Starting backend server..."
if ! pgrep -f "node server.js" > /dev/null; then
    nohup node server.js > server.log 2>&1 &
    echo "✅ Backend server started"
    sleep 3
else
    echo "✅ Backend server already running"
fi

echo ""
echo "🚀 Starting frontend portals..."

# Start all portals
start_portal 3001 admin "🔵" "Admin"
start_portal 3002 teacher "🟢" "Teacher"  
start_portal 3003 student "🟣" "Student"

echo ""
echo "⏳ Waiting for portals to start..."
sleep 15

echo ""
echo "✅ All services are now running!"
echo "==============================="
echo ""
echo "🔗 Portal URLs:"
echo "  🔵 Admin Portal:   http://localhost:3001"  
echo "  🟢 Teacher Portal: http://localhost:3002"
echo "  🟣 Student Portal: http://localhost:3003"
echo "  🔧 Backend API:    https://backend.edumetrix.uk"
echo ""
echo "⚡ Development Features:"
echo "  ✓ Auto-login enabled (no credentials needed)"
echo "  ✓ Mock data for all features"
echo "  ✓ All pages and routes functional"
echo "  ✓ Real-time updates"
echo ""
echo "📋 Development Users (auto-logged in):"
echo "  🔵 Admin:   Development Admin"
echo "  🟢 Teacher: Development Teacher"
echo "  🟣 Student: Development Student"
echo ""
echo "📝 Log files:"
echo "  Backend:     server.log"
echo "  Admin:       admin-portal.log"
echo "  Teacher:     teacher-portal.log"
echo "  Student:     student-portal.log"
echo ""
echo "🌐 Opening portals in browser..."

# Open URLs if possible
if command -v open > /dev/null; then
    # macOS
    sleep 2
    open http://localhost:3001 >/dev/null 2>&1 &
    sleep 1
    open http://localhost:3002 >/dev/null 2>&1 &
    sleep 1
    open http://localhost:3003 >/dev/null 2>&1 &
elif command -v xdg-open > /dev/null; then
    # Linux
    sleep 2
    xdg-open http://localhost:3001 >/dev/null 2>&1 &
    sleep 1
    xdg-open http://localhost:3002 >/dev/null 2>&1 &
    sleep 1
    xdg-open http://localhost:3003 >/dev/null 2>&1 &
fi

echo ""
echo "🎉 Development environment is ready!"
echo ""
echo "🛑 To stop all services:"
echo "   pkill -f 'PORT=300'"
echo "   pkill -f 'node server.js'"
echo ""
echo "🔄 To restart, just run this script again"
echo ""