import express from 'express';
import mongoose from 'mongoose';
import AdminSettings from '../../models/admin/AdminSettings.js';
import User from '../../models/admin/User.js';
import { adminAuth } from '../../middleware/auth.js';
import { validationResult, body } from 'express-validator';
import crypto from 'crypto';

const router = express.Router();

// Helper function to get valid adminId - handles both ObjectId and development string IDs
const getValidAdminId = (req) => {
  const adminId = req.user._id || req.user.id;

  // In development mode, we might have string IDs like 'dev-admin-123'
  // For the AdminSettings model to work, we need to either:
  // 1. Convert to ObjectId if it's a valid format, or
  // 2. Use the string as-is for development

  if (mongoose.Types.ObjectId.isValid(adminId)) {
    return adminId; // Valid ObjectId
  } else {
    // For development mode with string IDs, we'll convert to a valid ObjectId
    // by creating a consistent ObjectId from the string
    if (typeof adminId === 'string' && adminId.startsWith('dev-')) {
      // Create a consistent ObjectId from the dev string
      // We'll use the first 24 characters of a hash, padded if necessary
      const hash = crypto.createHash('md5').update(adminId).digest('hex');
      const objectIdHex = hash.substring(0, 24);
      return new mongoose.Types.ObjectId(objectIdHex);
    }
    return adminId; // Return as-is if not a dev string
  }
};

// Get current admin's settings
router.get('/', adminAuth, async (req, res) => {
  try {
    console.log('🔍 Admin Settings GET - req.user:', req.user);
    console.log('🔍 Admin Settings GET - req.user type:', typeof req.user);
    console.log('🔍 Admin Settings GET - req.user keys:', req.user ? Object.keys(req.user) : 'undefined');

    if (!req.user) {
      console.error('❌ Admin Settings GET - req.user is undefined');
      return res.status(401).json({
        success: false,
        message: 'User not authenticated'
      });
    }

    // For admins, req.user should have _id from the User document
    const adminId = getValidAdminId(req);

    let settings = await AdminSettings.findOne({ adminId });

    // If no settings exist, create default settings
    if (!settings) {
      const defaultSettings = AdminSettings.getDefaultSettings(adminId);
      settings = new AdminSettings(defaultSettings);
      await settings.save();
    }

    res.json({
      success: true,
      settings
    });
  } catch (error) {
    console.error('Error fetching admin settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch settings',
      error: error.message
    });
  }
});

// Update appearance settings
router.put('/appearance', [
  adminAuth,
  body('colorTheme').optional().isIn(['blue', 'purple', 'green', 'orange', 'pink', 'teal', 'cyan', 'indigo']),
  body('darkMode').optional().isBoolean()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const adminId = getValidAdminId(req);
    const { colorTheme, darkMode } = req.body;

    let settings = await AdminSettings.findOne({ adminId });

    if (!settings) {
      settings = new AdminSettings(AdminSettings.getDefaultSettings(adminId));
    }

    // Update appearance settings
    if (colorTheme !== undefined) settings.appearance.colorTheme = colorTheme;
    if (darkMode !== undefined) settings.appearance.darkMode = darkMode;

    await settings.save();

    res.json({
      success: true,
      message: 'Appearance settings updated successfully',
      settings: settings.appearance
    });
  } catch (error) {
    console.error('Error updating appearance settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update appearance settings',
      error: error.message
    });
  }
});

// Update notification settings
router.put('/notifications', [
  adminAuth,
  body('userRegistrations').optional().isBoolean(),
  body('systemAlerts').optional().isBoolean(),
  body('financialReports').optional().isBoolean(),
  body('classSchedules').optional().isBoolean(),
  body('trialRequests').optional().isBoolean(),
  body('messages').optional().isBoolean(),
  body('systemMaintenance').optional().isBoolean()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const adminId = getValidAdminId(req);
    const { userRegistrations, systemAlerts, financialReports, classSchedules, trialRequests, messages, systemMaintenance } = req.body;

    let settings = await AdminSettings.findOne({ adminId });

    if (!settings) {
      settings = new AdminSettings(AdminSettings.getDefaultSettings(adminId));
    }

    // Update notification settings
    if (userRegistrations !== undefined) settings.notifications.userRegistrations = userRegistrations;
    if (systemAlerts !== undefined) settings.notifications.systemAlerts = systemAlerts;
    if (financialReports !== undefined) settings.notifications.financialReports = financialReports;
    if (classSchedules !== undefined) settings.notifications.classSchedules = classSchedules;
    if (trialRequests !== undefined) settings.notifications.trialRequests = trialRequests;
    if (messages !== undefined) settings.notifications.messages = messages;
    if (systemMaintenance !== undefined) settings.notifications.systemMaintenance = systemMaintenance;

    await settings.save();

    res.json({
      success: true,
      message: 'Notification settings updated successfully',
      settings: settings.notifications
    });
  } catch (error) {
    console.error('Error updating notification settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update notification settings',
      error: error.message
    });
  }
});

// Update profile settings
router.put('/profile', [
  adminAuth,
  body('name').optional({ values: 'falsy' }).trim().isLength({ min: 1, max: 100 }),
  body('email').optional({ values: 'falsy' }).custom((value) => {
    if (value === '' || value == null) return true;
    return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value);
  }),
  body('phone').optional({ values: 'falsy' }).trim().isLength({ min: 0, max: 20 }),
  body('timezone').optional({ values: 'falsy' }).isString(),
  body('bio').optional({ values: 'falsy' }).trim().isLength({ min: 0, max: 500 }),
  body('jobTitle').optional({ values: 'falsy' }).trim().isLength({ min: 1, max: 100 })
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const adminId = getValidAdminId(req);
    const { name, email, phone, timezone, bio, jobTitle } = req.body;

    let settings = await AdminSettings.findOne({ adminId });

    if (!settings) {
      settings = new AdminSettings(AdminSettings.getDefaultSettings(adminId));
    }

    // Update profile settings
    if (name !== undefined) settings.profile.name = name;
    if (email !== undefined) settings.profile.email = email;
    if (phone !== undefined) settings.profile.phone = phone;
    if (timezone !== undefined) settings.profile.timezone = timezone;
    if (bio !== undefined) settings.profile.bio = bio;
    if (jobTitle !== undefined) settings.profile.jobTitle = jobTitle;

    await settings.save();

    // Also update the main admin user record if needed
    if (name || email) {
      const updateData = {};
      if (name) updateData.username = name;
      if (email) updateData.email = email;

      await User.findByIdAndUpdate(adminId, updateData);
    }

    res.json({
      success: true,
      message: 'Profile settings updated successfully',
      settings: settings.profile
    });
  } catch (error) {
    console.error('Error updating profile settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update profile settings',
      error: error.message
    });
  }
});

// Update privacy settings
router.put('/privacy', [
  adminAuth,
  body('dataAnalytics').optional().isBoolean(),
  body('promotionalEmails').optional().isBoolean(),
  body('profileVisibility').optional().isIn(['public', 'staff', 'private']),
  body('auditLogs').optional().isBoolean()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const adminId = getValidAdminId(req);
    const { dataAnalytics, promotionalEmails, profileVisibility, auditLogs } = req.body;

    let settings = await AdminSettings.findOne({ adminId });

    if (!settings) {
      settings = new AdminSettings(AdminSettings.getDefaultSettings(adminId));
    }

    // Update privacy settings
    if (dataAnalytics !== undefined) settings.privacy.dataAnalytics = dataAnalytics;
    if (promotionalEmails !== undefined) settings.privacy.promotionalEmails = promotionalEmails;
    if (profileVisibility !== undefined) settings.privacy.profileVisibility = profileVisibility;
    if (auditLogs !== undefined) settings.privacy.auditLogs = auditLogs;

    await settings.save();

    res.json({
      success: true,
      message: 'Privacy settings updated successfully',
      settings: settings.privacy
    });
  } catch (error) {
    console.error('Error updating privacy settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update privacy settings',
      error: error.message
    });
  }
});

// Update language settings
router.put('/language', [
  adminAuth,
  body('language').isIn(['en', 'es', 'fr', 'de', 'it', 'pt', 'zh', 'ja'])
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const adminId = getValidAdminId(req);
    const { language } = req.body;

    let settings = await AdminSettings.findOne({ adminId });

    if (!settings) {
      settings = new AdminSettings(AdminSettings.getDefaultSettings(adminId));
    }

    settings.language = language;
    await settings.save();

    res.json({
      success: true,
      message: 'Language settings updated successfully',
      language: settings.language
    });
  } catch (error) {
    console.error('Error updating language settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update language settings',
      error: error.message
    });
  }
});

// Update admin preferences
router.put('/admin-preferences', [
  adminAuth,
  body('autoApproveTrials').optional().isBoolean(),
  body('defaultClassDuration').optional().isInt({ min: 15, max: 180 }),
  body('emailDigests').optional().isBoolean(),
  body('systemBackups').optional().isBoolean(),
  body('dashboardLayout').optional().isIn(['compact', 'detailed', 'overview']),
  body('multiFactorAuth').optional().isBoolean()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const adminId = getValidAdminId(req);
    const { autoApproveTrials, defaultClassDuration, emailDigests, systemBackups, dashboardLayout, multiFactorAuth } = req.body;

    let settings = await AdminSettings.findOne({ adminId });

    if (!settings) {
      settings = new AdminSettings(AdminSettings.getDefaultSettings(adminId));
    }

    // Update admin preferences
    if (autoApproveTrials !== undefined) settings.adminPreferences.autoApproveTrials = autoApproveTrials;
    if (defaultClassDuration !== undefined) settings.adminPreferences.defaultClassDuration = defaultClassDuration;
    if (emailDigests !== undefined) settings.adminPreferences.emailDigests = emailDigests;
    if (systemBackups !== undefined) settings.adminPreferences.systemBackups = systemBackups;
    if (dashboardLayout !== undefined) settings.adminPreferences.dashboardLayout = dashboardLayout;
    if (multiFactorAuth !== undefined) settings.adminPreferences.multiFactorAuth = multiFactorAuth;

    await settings.save();

    res.json({
      success: true,
      message: 'Admin preferences updated successfully',
      settings: settings.adminPreferences
    });
  } catch (error) {
    console.error('Error updating admin preferences:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update admin preferences',
      error: error.message
    });
  }
});

// Update system settings
router.put('/system-settings', [
  adminAuth,
  body('maintenanceMode').optional().isBoolean(),
  body('maxStudentsPerClass').optional().isInt({ min: 1, max: 100 }),
  body('allowSelfRegistration').optional().isBoolean(),
  body('sessionTimeout').optional().isInt({ min: 30, max: 480 })
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    const adminId = getValidAdminId(req);
    const { maintenanceMode, maxStudentsPerClass, allowSelfRegistration, sessionTimeout } = req.body;

    let settings = await AdminSettings.findOne({ adminId });

    if (!settings) {
      settings = new AdminSettings(AdminSettings.getDefaultSettings(adminId));
    }

    // Update system settings
    if (maintenanceMode !== undefined) settings.systemSettings.maintenanceMode = maintenanceMode;
    if (maxStudentsPerClass !== undefined) settings.systemSettings.maxStudentsPerClass = maxStudentsPerClass;
    if (allowSelfRegistration !== undefined) settings.systemSettings.allowSelfRegistration = allowSelfRegistration;
    if (sessionTimeout !== undefined) settings.systemSettings.sessionTimeout = sessionTimeout;

    await settings.save();

    res.json({
      success: true,
      message: 'System settings updated successfully',
      settings: settings.systemSettings
    });
  } catch (error) {
    console.error('Error updating system settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update system settings',
      error: error.message
    });
  }
});

// Update all settings at once
router.put('/', [
  adminAuth,
  // Appearance validation
  body('appearance.colorTheme').optional().isIn(['blue', 'purple', 'green', 'orange', 'pink', 'teal', 'cyan', 'indigo']),
  body('appearance.darkMode').optional().isBoolean(),
  // Notification validation
  body('notifications.userRegistrations').optional().isBoolean(),
  body('notifications.systemAlerts').optional().isBoolean(),
  body('notifications.financialReports').optional().isBoolean(),
  body('notifications.classSchedules').optional().isBoolean(),
  body('notifications.trialRequests').optional().isBoolean(),
  body('notifications.messages').optional().isBoolean(),
  body('notifications.systemMaintenance').optional().isBoolean(),
  // Profile validation
  body('profile.name').optional({ values: 'falsy' }).trim().isLength({ min: 1, max: 100 }),
  body('profile.email').optional({ values: 'falsy' }).custom((value) => {
    if (value === '' || value == null) return true;
    return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value);
  }),
  body('profile.phone').optional({ values: 'falsy' }).trim().isLength({ min: 0, max: 20 }),
  body('profile.timezone').optional({ values: 'falsy' }).isString(),
  body('profile.bio').optional({ values: 'falsy' }).trim().isLength({ min: 0, max: 500 }),
  body('profile.jobTitle').optional({ values: 'falsy' }).trim().isLength({ min: 1, max: 100 }),
  // Privacy validation
  body('privacy.dataAnalytics').optional().isBoolean(),
  body('privacy.promotionalEmails').optional().isBoolean(),
  body('privacy.profileVisibility').optional().isIn(['public', 'staff', 'private']),
  body('privacy.auditLogs').optional().isBoolean(),
  // Language validation
  body('language').optional().isIn(['en', 'es', 'fr', 'de', 'it', 'pt', 'zh', 'ja']),
  // Admin preferences validation
  body('adminPreferences.autoApproveTrials').optional().isBoolean(),
  body('adminPreferences.defaultClassDuration').optional().isInt({ min: 15, max: 180 }),
  body('adminPreferences.emailDigests').optional().isBoolean(),
  body('adminPreferences.systemBackups').optional().isBoolean(),
  body('adminPreferences.dashboardLayout').optional().isIn(['compact', 'detailed', 'overview']),
  body('adminPreferences.multiFactorAuth').optional().isBoolean(),
  // System settings validation
  body('systemSettings.maintenanceMode').optional().isBoolean(),
  body('systemSettings.maxStudentsPerClass').optional().isInt({ min: 1, max: 100 }),
  body('systemSettings.allowSelfRegistration').optional().isBoolean(),
  body('systemSettings.sessionTimeout').optional().isInt({ min: 30, max: 480 })
], async (req, res) => {
  try {
    console.log('💾 Admin Settings PUT BULK - req.user:', req.user);
    console.log('💾 Admin Settings PUT BULK - req.body:', req.body);

    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Validation failed',
        errors: errors.array()
      });
    }

    if (!req.user) {
      console.error('❌ Admin Settings PUT BULK - req.user is undefined');
      return res.status(401).json({
        success: false,
        message: 'User not authenticated'
      });
    }

    // For admins, req.user should have _id from the User document
    const adminId = getValidAdminId(req);
    const { appearance, notifications, profile, privacy, language, adminPreferences, systemSettings } = req.body;

    let settings = await AdminSettings.findOne({ adminId });

    if (!settings) {
      settings = new AdminSettings(AdminSettings.getDefaultSettings(adminId));
    }

    // Update all categories
    if (appearance) {
      Object.assign(settings.appearance, appearance);
    }
    if (notifications) {
      Object.assign(settings.notifications, notifications);
    }
    if (profile) {
      Object.assign(settings.profile, profile);
    }
    if (privacy) {
      Object.assign(settings.privacy, privacy);
    }
    if (language) {
      settings.language = language;
    }
    if (adminPreferences) {
      Object.assign(settings.adminPreferences, adminPreferences);
    }
    if (systemSettings) {
      Object.assign(settings.systemSettings, systemSettings);
    }

    await settings.save();

    // Update main admin user record if profile data changed
    if (profile && (profile.name || profile.email)) {
      const updateData = {};
      if (profile.name) updateData.username = profile.name;
      if (profile.email) updateData.email = profile.email;

      await User.findByIdAndUpdate(adminId, updateData);
    }

    res.json({
      success: true,
      message: 'Settings updated successfully',
      settings
    });
  } catch (error) {
    console.error('Error updating settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update settings',
      error: error.message
    });
  }
});

// Reset settings to default
router.post('/reset', adminAuth, async (req, res) => {
  try {
    const adminId = getValidAdminId(req);

    // Delete existing settings
    await AdminSettings.findOneAndDelete({ adminId });

    // Create new default settings
    const defaultSettings = AdminSettings.getDefaultSettings(adminId);
    const newSettings = new AdminSettings(defaultSettings);
    await newSettings.save();

    res.json({
      success: true,
      message: 'Settings reset to default successfully',
      settings: newSettings
    });
  } catch (error) {
    console.error('Error resetting settings:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to reset settings',
      error: error.message
    });
  }
});

export default router;