import express from 'express';
import Message from '../../models/shared/Message.js';
import Employee from '../../models/teacher/Teacher.js';
import mongoose from 'mongoose';
import multer from 'multer';
import path from 'path';

import Student from '../../models/student/Student.js';
import Admin from '../../models/admin/User.js';
import Conversation from '../../models/shared/Conversation.js';

const router = express.Router();

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, 'uploads/chat/');
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({
  storage,
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB limit
  fileFilter: (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif|pdf|doc|docx|xls|xlsx|txt|mp4|mp3|wav/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Invalid file type. Only images, documents, videos, and audio files are allowed.'));
    }
  }
});

export default function messageRouter(io, users){
  // ✅ Fetch only broadcast messages for teachers
  router.get('/teacher/broadcast-messages', async (req, res) => {
    try {
      // Fetch only group messages sent to teachers (handle both 'Employee' and 'Teacher')
      const broadcastMessages = await Message.find({
        receiverType: { $in: ['Employee', 'Teacher'] },
        isGroupMessage: true,
      }).sort({ timestamp: 1 });

      res.status(200).json({ broadcastMessages });
    } catch (err) {
      console.error('Error fetching broadcast messages:', err);
      res.status(500).json({ error: err.message });
    }
  });
  router.get('/student/broadcast-messages', async (req, res) => {
    try {
      // Fetch only group messages sent to teachers
      const broadcastMessages = await Message.find({
        receiverType: 'Student',
        isGroupMessage: true,
      }).sort({ timestamp: 1 });

      res.status(200).json({ broadcastMessages });
    } catch (err) {
      console.error('Error fetching broadcast messages:', err);
      res.status(500).json({ error: err.message });
    }
  });
  //get all students
  router.get('/student/:teacherId', async (req, res) => {
    const { teacherId } = req.params;

    try {
      const latestMessages = await Message.aggregate([
        {
          $match: {
            $or: [
              { senderType: 'Student', receiverId: new mongoose.Types.ObjectId(teacherId) },
              { receiverType: 'Student', senderId: new mongoose.Types.ObjectId(teacherId) },
            ],
          },
        },
        { $sort: { timestamp: -1 } },
        {
          $group: {
            _id: {
              $cond: [{ $eq: ['$senderType', 'Student'] }, '$senderId', '$receiverId'],
            },
            lastMessage: { $first: '$content' },
            timestamp: { $first: '$timestamp' },
          },
        },
      ]);

      const students = await Student.find({ teacher: teacherId })
        .populate('class', 'className year subjectName')
        .populate('teacher', 'employeeName')
        .lean();

      const latestMap = Object.fromEntries(latestMessages.map((msg) => [msg._id.toString(), msg]));

      students.forEach((student) => {
        const msgData = latestMap[student._id.toString()];
        student.timestamp = msgData?.timestamp || null;
        student.lastMessage = msgData?.lastMessage || null;

        if (student.picture?.data) {
          student.picture = `data:${student.picture.contentType};base64,${Buffer.from(student.picture.data).toString(
            'base64'
          )}`;
        } else {
          student.picture = null;
        }
      });

      // Sort by most recent message timestamp
      students.sort((a, b) => new Date(b.timestamp || 0) - new Date(a.timestamp || 0));

      res.status(200).json(students);
    } catch (error) {
      res.status(500).json({ message: 'Failed to fetch students', error: error.message });
    }
  });

  router.get('/employees/:id', async (req, res) => {
    const { id } = req.params;

    try {
      const student = await Student.findById(id).populate('teacher', '_id employeeName');

      if (!student) {
        return res.status(404).json({ error: 'Student not found' });
      }

      const teacherIds = student.teacher.map((t) => t._id);

      const latestMessages = await Message.aggregate([
        {
          $match: {
            $or: [
              { senderId: new mongoose.Types.ObjectId(id), receiverId: { $in: teacherIds } },
              { receiverId: new mongoose.Types.ObjectId(id), senderId: { $in: teacherIds } },
            ],
          },
        },
        { $sort: { timestamp: -1 } },
        {
          $group: {
            _id: {
              $cond: [{ $eq: ['$senderId', new mongoose.Types.ObjectId(id)] }, '$receiverId', '$senderId'],
            },
            lastMessage: { $first: '$content' },
            timestamp: { $first: '$timestamp' },
          },
        },
      ]);

      const latestMap = Object.fromEntries(latestMessages.map((msg) => [msg._id.toString(), msg]));

      const teacherDetails = student.teacher.map((teacher) => {
        const msgData = latestMap[teacher._id.toString()];
        return {
          _id: teacher._id,
          name: teacher.employeeName,
          lastMessage: msgData?.lastMessage || null,
          timestamp: msgData?.timestamp || null,
        };
      });

      teacherDetails.sort((a, b) => new Date(b.timestamp || 0) - new Date(a.timestamp || 0));

      res.json(teacherDetails);
    } catch (err) {
      res.status(500).json({ error: err.message });
    }
  });

  // Get Admin details

  router.get('/admin', async (req, res) => {
  const { userId, userType } = req.query; // e.g. userType = 'Employee' or 'Student'

  try {
    const admin = await Admin.findOne().select('_id name').lean();

    if (!admin) {
      return res.status(404).json({ error: 'Admin not found' });
    }

    // Get the latest message between the admin and the given user
    const lastMessage = await Message.findOne({
      $or: [
        {
          senderId: admin._id.toString(),
          senderType: 'Admin',
          receiverId: userId,
          receiverType: userType,
        },
        {
          receiverId: admin._id.toString(),
          receiverType: 'Admin',
          senderId: userId,
          senderType: userType,
        },
      ],
    })
      .sort({ timestamp: -1 })
      .lean();

    const enrichedAdmin = {
      ...admin,
      lastMessage: lastMessage?.content || '',
      timestamp: lastMessage?.timestamp || null,
    };

    res.json([enrichedAdmin]);
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});


  // Get a specific conversation by conversationId
  router.get('/messages/:conversationId', async (req, res) => {
    const { conversationId } = req.params;

    try {
      const messages = await Message.find({ conversationId }).sort({ timestamp: 1 });
      res.json(messages);
    } catch (err) {
      res.status(500).json({ error: err.message });
    }
  });

  router.post('/messages', async (req, res) => {
    const { senderId, senderType, receiverId, receiverType, content, conversationId, timestamp } = req.body;

    try {
      // Check if sender is a student
      const sender = await Employee.findById(senderId);
      if (!sender) {
        return res.status(403).json({ error: 'Only employees can send messages' });
      }

      // Ensure receiver is an student
      const receiver = (await Student.findById(receiverId)) || (await Admin.findById(receiverId));

      if (!receiver) {
        return res.status(404).json({ error: 'Receiver not found' });
      }

      // Create new message document
      const message = new Message({
        senderId,
        senderType,
        receiverId,
        receiverType,
        conversationId,
        content,
        timestamp: new Date(timestamp),
      });

      // Save message to DB
      await message.save();

      const room = conversationId; // Ensure users are in the same room for communication
      io.to(room).emit('new_message', message);

      // Send response back to frontend
      res.status(200).json({ message: 'Message sent successfully!' });

      // Emit to the conversation (using Socket.IO)
      // io.to(conversationId).emit('new_message', message); // Emit to the specific conversation room
    } catch (err) {
      console.error('Error saving message:', err.message);
      res.status(500).json({ error: err.message });
    }
  });

  // Send a message (Only Students can message Employees)
  router.post('/messages', async (req, res) => {
    const { senderId, receiverId, receiverType, content, conversationId, timestamp } = req.body;

    try {
      // Check if sender is a student
      const sender = await Student.findById(senderId);
      if (!sender) {
        return res.status(403).json({ error: 'Only students can send messages' });
      }

      // Ensure receiver is an employee
      const receiver = (await Employee.findById(receiverId)) || (await Admin.findById(receiverId));

      if (!receiver) {
        return res.status(404).json({ error: 'Receiver not found' });
      }

      // Create new message document
      const message = new Message({
        senderId,
        receiverId,
        receiverType, // Should be 'Employee'
        conversationId,
        content,
        timestamp: new Date(timestamp),
      });

      // Save message to DB
      await message.save();

      // Send response back to frontend
      res.status(200).json({ message: 'Message sent successfully!' });

      // Emit to the conversation (using Socket.IO)
      // io.to(conversationId).emit('new_message', message); // Emit to the specific conversation room
    } catch (err) {
      console.error('Error saving message:', err.message);
      res.status(500).json({ error: err.message });
    }
  });

  //teacher-message-send
  router.post('/teacher/message-send', upload.single('attachment'), async (req, res) => {
    try {
      const { senderId, receiverId, receiverType, content } = req.body;

      // Validate sender (teacher)
      const teacher = await Employee.findById(senderId);
      if (!teacher) return res.status(404).json({ error: 'Teacher not found' });

      // Validate receiver (Admin or Student)
      let receiver;
      if (receiverType === 'Admin') {
        receiver = await Admin.findById(receiverId);
      } else if (receiverType === 'Student') {
        receiver = await Student.findById(receiverId);
      }

      if (!receiver) return res.status(404).json({ error: 'Receiver not found' });

      // Check if a conversation exists
      let conversation = await Conversation.findOne({
        participants: {
          $all: [
            { userId: senderId, userType: 'Teacher' },
            { userId: receiverId, userType: receiverType },
          ],
        },
      });

      // If no conversation exists, create one
      if (!conversation) {
        conversation = new Conversation({
          participants: [
            { userId: senderId, userType: 'Teacher' },
            { userId: receiverId, userType: receiverType },
          ],
          isGroup: false,
          isVisibleToAdmin: true, // ✅ Allow admin to see
        });
        await conversation.save();
      }

      // Prepare attachment data if file was uploaded
      let attachment = null;
      if (req.file) {
        attachment = {
          fileName: req.file.originalname,
          fileUrl: `/uploads/chat/${req.file.filename}`,
          fileType: req.file.mimetype,
          fileSize: req.file.size
        };
      }

      // Create a new message
      const message = new Message({
        conversationId: conversation._id,
        senderId,
        senderType: 'Teacher',
        receiverId,
        receiverType,
        content,
        attachment,
        timestamp: new Date(),
      });

      await message.save();

      // Emit socket event for real-time delivery
      const conversationId = conversation._id.toString();

      console.log('🔔 Emitting teacher message to conversation:', conversationId);
      console.log('🔔 Message data:', message);
      console.log('🔔 Receiver ID:', receiverId);
      console.log('🔔 Online users:', Object.keys(users));

      // Emit to conversation room
      io.to(conversationId).emit('receive_message', message);
      console.log(`📡 Emitted to conversation room: ${conversationId}`);

      // Also emit directly to receiver's socket if they're online
      if (users[receiverId]) {
        users[receiverId].forEach((socketId) => {
          io.to(socketId).emit('receive_message', message);
          console.log(`📡 Emitted to socket: ${socketId}`);
        });
        console.log(`📨 Teacher message delivered to ${receiverType} ${receiverId} in real-time`);
      } else {
        console.log(`⚠️ Receiver ${receiverId} is offline. Online users:`, Object.keys(users));
      }

      // Send notification to receiver
      const { createNotification, NotificationTemplates } = await import('../../utils/notificationHelper.js');
      const notificationData = NotificationTemplates.messageReceived(teacher.name || 'Teacher');

      await createNotification({
        userId: receiverId,
        userType: receiverType.toLowerCase(),
        type: notificationData.type,
        title: notificationData.title,
        message: notificationData.message,
        relatedId: message._id,
        relatedModel: 'Message',
        priority: notificationData.priority,
        actionUrl: receiverType === 'Admin' ? '/admin/chats' : '/student/chat',
        io
      });

      res.status(200).json({
        message: 'Message sent successfully',
        data: message,
        conversationId: conversation._id, // ✅ Now frontend can use this!
      });
    } catch (err) {
      console.error('Error sending message:', err);
      res.status(500).json({ error: err.message });
    }
  });

  router.get('/teacher/get-messages/:teacherId/:receiverId', async (req, res) => {
    try {
      const { teacherId, receiverId } = req.params;

      if (!teacherId || !receiverId) {
        return res.status(400).json({ error: 'Teacher ID and Receiver ID are required' });
      }

      const messages = await Message.find({
        $or: [
          { senderId: teacherId, senderType: 'Teacher', receiverId: receiverId },
          { senderId: receiverId, receiverType: 'Teacher', receiverId: teacherId },
        ],
        isGroupMessage: false,
      }).sort({ timestamp: 1 });

      res.status(200).json({ messages });
    } catch (err) {
      console.error('Error fetching messages:', err);
      res.status(500).json({ error: err.message });
    }
  });

  //student message

  router.post('/student/message-send', upload.single('attachment'), async (req, res) => {
    try {
      const { senderId, receiverId, receiverType, content } = req.body;

      console.log('req body>>',req.body)
      console.log('req file>>',req.file)

      // Validate sender (student)
      const student = await Student.findById(new mongoose.Types.ObjectId(senderId));
      console.log('student>>',student)
      if (!student) return res.status(404).json({ error: 'Student not found' });

      // Validate receiver (Admin or Teacher)
      let receiver;
      if (receiverType === 'Admin') {
        receiver = await Admin.findById(receiverId);
      } else if (receiverType === 'Teacher') {
        receiver = await Employee.findById(receiverId);
      }

      if (!receiver) return res.status(404).json({ error: 'Receiver not found' });

      // Check if a conversation exists
      let conversation = await Conversation.findOne({
        participants: {
          $all: [
            { userId: senderId, userType: 'Student' },
            { userId: receiverId, userType: receiverType },
          ],
        },
      });

      // If no conversation exists, create one
      if (!conversation) {
        conversation = new Conversation({
          participants: [
            { userId: senderId, userType: 'Student' },
            { userId: receiverId, userType: receiverType },
          ],
          isGroup: false,
          isVisibleToAdmin: true, // ✅
        });
        await conversation.save();
      }

      // Prepare attachment data if file was uploaded
      let attachment = null;
      if (req.file) {
        attachment = {
          fileName: req.file.originalname,
          fileUrl: `/uploads/chat/${req.file.filename}`,
          fileType: req.file.mimetype,
          fileSize: req.file.size
        };
      }

      // Create a new message
      const message = new Message({
        conversationId: conversation._id,
        senderId,
        senderType: 'Student',
        receiverId,
        receiverType,
        content,
        attachment,
        timestamp: new Date(),
      });

      await message.save();

      // Emit socket event for real-time delivery
      const conversationId = conversation._id.toString();

      console.log('🔔 Emitting student message to conversation:', conversationId);
      console.log('🔔 Message data:', message);
      console.log('🔔 Receiver ID:', receiverId);
      console.log('🔔 Online users:', Object.keys(users));

      // Emit to conversation room
      io.to(conversationId).emit('receive_message', message);
      console.log(`📡 Emitted to conversation room: ${conversationId}`);

      // Also emit directly to receiver's socket if they're online
      if (users[receiverId]) {
        users[receiverId].forEach((socketId) => {
          io.to(socketId).emit('receive_message', message);
          console.log(`📡 Emitted to socket: ${socketId}`);
        });
        console.log(`📨 Student message delivered to ${receiverType} ${receiverId} in real-time`);
      } else {
        console.log(`⚠️ Receiver ${receiverId} is offline. Online users:`, Object.keys(users));
      }

      // Send notification to receiver
      const { createNotification, NotificationTemplates } = await import('../../utils/notificationHelper.js');
      const notificationData = NotificationTemplates.messageReceived(student.studentName || 'Student');

      await createNotification({
        userId: receiverId,
        userType: receiverType.toLowerCase(),
        type: notificationData.type,
        title: notificationData.title,
        message: notificationData.message,
        relatedId: message._id,
        relatedModel: 'Message',
        priority: notificationData.priority,
        actionUrl: receiverType === 'Admin' ? '/admin/chats' : '/teacher/chat',
        io
      });

      res.status(200).json({ message: 'Message sent successfully', data: message });
    } catch (err) {
      console.error('Error sending message:', err);
      res.status(500).json({ error: err.message });
    }
  });

  router.get('/student/get-messages/:studentId/:receiverId', async (req, res) => {
    try {
      const { studentId, receiverId } = req.params;

      if (!studentId || !receiverId) {
        return res.status(400).json({ error: 'Teacher ID and Receiver ID are required' });
      }

      const messages = await Message.find({
        $or: [
          { senderId: studentId, senderType: 'Student', receiverId: receiverId },
          { senderId: receiverId, receiverType: 'Student', receiverId: studentId },
        ],
        isGroupMessage: false,
      }).sort({ timestamp: 1 });

      res.status(200).json({ messages });
    } catch (err) {
      console.error('Error fetching messages:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Get conversation list for student with last message and unread count
  router.get('/student/conversations/:studentId', async (req, res) => {
    try {
      const { studentId } = req.params;

      if (!studentId) {
        return res.status(400).json({ error: 'Student ID is required' });
      }

      // Get all conversations for this student
      const conversations = await Conversation.find({
        'participants.userId': studentId,
        'participants.userType': 'Student',
      }).lean();

      // Get conversation details with last message and unread count
      const conversationDetails = await Promise.all(
        conversations.map(async (conv) => {
          // Find the other participant (teacher or admin)
          const otherParticipant = conv.participants.find(
            (p) => p.userId.toString() !== studentId
          );

          if (!otherParticipant) return null;

          // Get the last message - fetch multiple to debug
          const allMessages = await Message.find({
            conversationId: conv._id.toString(),
          })
            .sort({ timestamp: -1 })
            .limit(3)
            .lean();

          console.log(`📧 Student Conversation - ConvID: ${conv._id.toString()}, User: ${otherParticipant.userId}`);
          console.log(`   Found ${allMessages.length} messages (showing last 3):`);
          allMessages.forEach((msg, idx) => {
            console.log(`   ${idx + 1}. "${msg.content}" - ${msg.timestamp} - Sender: ${msg.senderType}`);
          });

          const lastMessage = allMessages[0];

          // Get unread count (messages from other participant that student hasn't read)
          const unreadCount = await Message.countDocuments({
            conversationId: conv._id.toString(),
            senderId: otherParticipant.userId,
            senderType: otherParticipant.userType,
            isRead: false,
          });

          return {
            conversationId: conv._id,
            userId: otherParticipant.userId,
            userType: otherParticipant.userType,
            lastMessage: lastMessage?.attachment ? '📎 Attachment' : (lastMessage?.content || null),
            lastMessageTime: lastMessage?.timestamp || conv.updatedAt,
            unreadCount,
          };
        })
      );

      // Filter out null entries and sort by most recent message
      const validConversations = conversationDetails
        .filter((c) => c !== null)
        .sort((a, b) => new Date(b.lastMessageTime) - new Date(a.lastMessageTime));

      res.status(200).json({ conversations: validConversations });
    } catch (err) {
      console.error('Error fetching conversations:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Mark messages as read when student opens conversation
  router.post('/student/mark-read', async (req, res) => {
    try {
      const { studentId, conversationId, receiverId } = req.body;

      if (!studentId) {
        return res.status(400).json({ error: 'Student ID is required' });
      }

      let query = {
        receiverId: studentId,
        receiverType: 'Student',
        isRead: false,
      };

      // If conversationId is provided, use it
      if (conversationId) {
        query.conversationId = conversationId;
      } else if (receiverId) {
        // Fallback: use senderId if conversationId is not available
        query.senderId = receiverId;
      } else {
        return res.status(400).json({ error: 'Either conversationId or receiverId is required' });
      }

      // Mark all messages in this conversation from other participants as read
      const result = await Message.updateMany(query, {
        $set: {
          isRead: true,
          readAt: new Date(),
        },
      });

      console.log('✅ Marked messages as read:', result.modifiedCount);

      res.status(200).json({
        message: 'Messages marked as read',
        modifiedCount: result.modifiedCount,
      });
    } catch (err) {
      console.error('Error marking messages as read:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Get conversation list for teacher with last message and unread count
  router.get('/teacher/conversations/:teacherId', async (req, res) => {
    try {
      const { teacherId } = req.params;

      if (!teacherId) {
        return res.status(400).json({ error: 'Teacher ID is required' });
      }

      // Get all conversations for this teacher
      const conversations = await Conversation.find({
        'participants.userId': teacherId,
        'participants.userType': 'Teacher',
      }).lean();

      // Get conversation details with last message and unread count
      const conversationDetails = await Promise.all(
        conversations.map(async (conv) => {
          // Find the other participant (student or admin)
          const otherParticipant = conv.participants.find(
            (p) => p.userId.toString() !== teacherId
          );

          if (!otherParticipant) return null;

          // Get the last message - fetch multiple to debug
          const allMessages = await Message.find({
            conversationId: conv._id.toString(),
          })
            .sort({ timestamp: -1 })
            .limit(3)
            .lean();

          console.log(`📧 Teacher Conversation - ConvID: ${conv._id.toString()}, User: ${otherParticipant.userId}`);
          console.log(`   Found ${allMessages.length} messages (showing last 3):`);
          allMessages.forEach((msg, idx) => {
            console.log(`   ${idx + 1}. "${msg.content}" - ${msg.timestamp} - Sender: ${msg.senderType}`);
          });

          const lastMessage = allMessages[0];

          // Get unread count (messages from other participant that teacher hasn't read)
          const unreadCount = await Message.countDocuments({
            conversationId: conv._id.toString(),
            senderId: otherParticipant.userId,
            senderType: otherParticipant.userType,
            isRead: false,
          });

          return {
            conversationId: conv._id,
            userId: otherParticipant.userId,
            userType: otherParticipant.userType,
            lastMessage: lastMessage?.attachment ? '📎 Attachment' : (lastMessage?.content || null),
            lastMessageTime: lastMessage?.timestamp || conv.updatedAt,
            unreadCount,
          };
        })
      );

      // Filter out null entries and sort by most recent message
      const validConversations = conversationDetails
        .filter((c) => c !== null)
        .sort((a, b) => new Date(b.lastMessageTime) - new Date(a.lastMessageTime));

      res.status(200).json({ conversations: validConversations });
    } catch (err) {
      console.error('Error fetching teacher conversations:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Mark messages as read when teacher opens conversation
  router.post('/teacher/mark-read', async (req, res) => {
    try {
      const { teacherId, conversationId, receiverId } = req.body;

      if (!teacherId) {
        return res.status(400).json({ error: 'Teacher ID is required' });
      }

      let query = {
        receiverId: teacherId,
        receiverType: 'Teacher',
        isRead: false,
      };

      // If conversationId is provided, use it
      if (conversationId) {
        query.conversationId = conversationId;
      } else if (receiverId) {
        // Fallback: use senderId if conversationId is not available
        query.senderId = receiverId;
      } else {
        return res.status(400).json({ error: 'Either conversationId or receiverId is required' });
      }

      // Mark all messages in this conversation from other participants as read
      const result = await Message.updateMany(query, {
        $set: {
          isRead: true,
          readAt: new Date(),
        },
      });

      console.log('✅ Marked messages as read:', result.modifiedCount);

      res.status(200).json({
        message: 'Messages marked as read',
        modifiedCount: result.modifiedCount,
      });
    } catch (err) {
      console.error('Error marking messages as read:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Get conversation list for admin with last message and unread count
  router.get('/admin/conversations/:adminId', async (req, res) => {
    try {
      const { adminId } = req.params;

      if (!adminId) {
        return res.status(400).json({ error: 'Admin ID is required' });
      }

      // Get all conversations for this admin
      const conversations = await Conversation.find({
        'participants.userId': adminId,
        'participants.userType': 'Admin',
      }).lean();

      // Get conversation details with last message and unread count
      const conversationDetails = await Promise.all(
        conversations.map(async (conv) => {
          // Find the other participant (student or teacher)
          const otherParticipant = conv.participants.find(
            (p) => p.userId.toString() !== adminId
          );

          if (!otherParticipant) return null;

          // Get the last message
          const lastMessage = await Message.findOne({
            conversationId: conv._id.toString(),
          })
            .sort({ timestamp: -1 })
            .lean();

          // Get unread count (messages from other participant that admin hasn't read)
          const unreadCount = await Message.countDocuments({
            conversationId: conv._id.toString(),
            senderId: otherParticipant.userId,
            senderType: otherParticipant.userType,
            isRead: false,
          });

          return {
            conversationId: conv._id,
            userId: otherParticipant.userId,
            userType: otherParticipant.userType,
            lastMessage: lastMessage?.attachment ? '📎 Attachment' : (lastMessage?.content || null),
            lastMessageTime: lastMessage?.timestamp || conv.updatedAt,
            unreadCount,
          };
        })
      );

      // Filter out null entries and sort by most recent message
      const validConversations = conversationDetails
        .filter((c) => c !== null)
        .sort((a, b) => new Date(b.lastMessageTime) - new Date(a.lastMessageTime));

      res.status(200).json({ conversations: validConversations });
    } catch (err) {
      console.error('Error fetching admin conversations:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Mark messages as read when admin opens conversation
  router.post('/admin/mark-read', async (req, res) => {
    try {
      const { adminId, conversationId, receiverId } = req.body;

      if (!adminId) {
        return res.status(400).json({ error: 'Admin ID is required' });
      }

      let query = {
        receiverId: adminId,
        receiverType: 'Admin',
        isRead: false,
      };

      // If conversationId is provided, use it
      if (conversationId) {
        query.conversationId = conversationId;
      } else if (receiverId) {
        // Fallback: use senderId if conversationId is not available
        query.senderId = receiverId;
      } else {
        return res.status(400).json({ error: 'Either conversationId or receiverId is required' });
      }

      // Mark all messages in this conversation from other participants as read
      const result = await Message.updateMany(query, {
        $set: {
          isRead: true,
          readAt: new Date(),
        },
      });

      console.log('✅ Marked messages as read:', result.modifiedCount);

      res.status(200).json({
        message: 'Messages marked as read',
        modifiedCount: result.modifiedCount,
      });
    } catch (err) {
      console.error('Error marking messages as read:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Get all teacher-student conversations for admin monitoring
  router.get('/admin/all-conversations', async (req, res) => {
    try {
      // Find all messages between teachers and students
      const allMessages = await Message.find({
        $or: [
          { senderType: 'Teacher', receiverType: 'Student' },
          { senderType: 'Student', receiverType: 'Teacher' },
        ],
      })
        .sort({ timestamp: -1 })
        .lean();

      // Group messages by teacher-student pair
      const conversationMap = new Map();

      for (const message of allMessages) {
        let teacherId, studentId;

        // Determine teacher and student IDs
        if (message.senderType === 'Teacher') {
          teacherId = message.senderId;
          studentId = message.receiverId;
        } else {
          teacherId = message.receiverId;
          studentId = message.senderId;
        }

        // Create a unique key for this teacher-student pair
        const pairKey = `${teacherId.toString()}_${studentId.toString()}`;

        // If this pair doesn't exist yet, create it
        if (!conversationMap.has(pairKey)) {
          conversationMap.set(pairKey, {
            teacherId,
            studentId,
            lastMessage: message.content,
            lastMessageTime: message.timestamp,
            lastMessageSender: message.senderType,
            conversationId: message.conversationId,
            messageCount: 0,
          });
        }

        // Increment message count for this pair
        conversationMap.get(pairKey).messageCount++;
      }

      // Get teacher and student details for each conversation
      const conversationDetails = await Promise.all(
        Array.from(conversationMap.values()).map(async (conv) => {
          // Get teacher details
          const teacherDoc = await Employee.findById(conv.teacherId).select('name email').lean();

          // Get student details
          const studentDoc = await Student.findById(conv.studentId).select('studentName email').lean();

          if (!teacherDoc || !studentDoc) return null;

          // Get the actual last message to check for attachment
          const lastMsg = await Message.findOne({
            conversationId: conv.conversationId,
          })
            .sort({ timestamp: -1 })
            .lean();

          return {
            teacherId: conv.teacherId,
            studentId: conv.studentId,
            teacher: {
              _id: conv.teacherId,
              name: teacherDoc.name || 'Unknown Teacher',
              email: teacherDoc.email,
            },
            student: {
              _id: conv.studentId,
              name: studentDoc.studentName || 'Unknown Student',
              email: studentDoc.email,
            },
            lastMessage: lastMsg?.attachment ? '📎 Attachment' : (conv.lastMessage || null),
            lastMessageTime: conv.lastMessageTime,
            lastMessageSender: conv.lastMessageSender,
            messageCount: conv.messageCount,
            conversationId: conv.conversationId,
          };
        })
      );

      // Filter out null entries and sort by most recent message
      const validConversations = conversationDetails
        .filter((c) => c !== null)
        .sort((a, b) => new Date(b.lastMessageTime) - new Date(a.lastMessageTime));

      res.status(200).json({ conversations: validConversations });
    } catch (err) {
      console.error('Error fetching all conversations:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // Get messages for a specific teacher-student pair (for admin viewing)
  router.get('/admin/conversation-messages/:teacherId/:studentId', async (req, res) => {
    try {
      const { teacherId, studentId } = req.params;

      if (!teacherId || !studentId) {
        return res.status(400).json({ error: 'Teacher ID and Student ID are required' });
      }

      // Get all messages between this teacher and student (regardless of conversation)
      const messages = await Message.find({
        $or: [
          { senderId: teacherId, receiverId: studentId },
          { senderId: studentId, receiverId: teacherId },
        ],
      })
        .sort({ timestamp: 1 })
        .lean();

      res.status(200).json({ messages });
    } catch (err) {
      console.error('Error fetching conversation messages:', err);
      res.status(500).json({ error: err.message });
    }
  });

  // ✅ Fetch only broadcast messages for teachers

  return router;
}
