import express from "express";
import mongoose from "mongoose";
import dotenv from "dotenv";
import http from "http";
import { Server } from "socket.io";
import fs from "fs";
import path from "path";
import jwt from "jsonwebtoken";

dotenv.config();

import messageRouter from "./routes/shared/message.js";
import adminMessageRoutes from "./routes/admin/message.js";
// Load environment variables first

// Import security middleware
import {
  helmetConfig,
  corsOptions,
  generalLimiter,
  authLimiter,
  otpLimiter,
  strictLimiter,
  bruteForceProtection,
  authBruteForce,
  validateInput,
  validationSchemas,
  xssProtection,
  mongoSanitizer,
  securityAudit,
  adminRouteProtection,
  requestSizeLimiter,
  securityLogger,
  hppMiddleware as hpp
} from './middleware/security.js';

import cors from 'cors';

import { auth, requireAdmin } from './middleware/advancedAuth.js';
import authRoutes from "./routes/admin/auth.js";
import adminRoutes from "./routes/admin/admin.js";
import adminExamsRoutes from "./routes/admin/exams.js";
import adminSettingsRoutes from "./routes/admin/settings.js";

import teacherRoutes from "./routes/teacher/teacher.js";
import teacherExamsRoutes from "./routes/teacher/exams.js";
import teacherAssignmentsRoutes from "./routes/teacher/assignments.js";
import teacherQuicktestRoutes from "./routes/teacher/quicktest.js";
import teacherHomeworkRoutes from "./routes/teacher/homework.js";
import teacherSettingsRoutes from "./routes/teacher/settings.js";
import teacherAttendanceRoutes from "./routes/teacher/attendance.js";
import teacherReportsRoutes from "./routes/teacher/reports.js";

import studentRoutes from "./routes/student/student.js";
import studentExamsRoutes from "./routes/student/exams.js";
import studentAssignmentsRoutes from "./routes/student/assignments.js";
import studentQuicktestRoutes from "./routes/student/quicktest.js";
import studentHomeworkRoutes from "./routes/student/homework.js";
import studentSettingsRoutes from "./routes/student/settings.js";
import studentAttendanceRoutes from "./routes/student/attendance.js";

import chatRoutes from "./routes/admin/chat.js";
import classesRoutes from "./routes/teacher/classes.js";
import financeRoutes from "./routes/admin/finance.js";
import reportsRoutes from "./routes/admin/reports.js";
import commonRoutes from './routes/shared/common.js';
import notificationRoutes from './routes/shared/notifications.js';
import testNotificationRoutes from './routes/shared/testNotification.js';
// Ensure logs directory exists
if (!fs.existsSync('./logs')) {
  fs.mkdirSync('./logs');
}

const app = express();
const server = http.createServer(app);

// Configure Express to trust proxy for development
// app.set('trust proxy', false);
app.set('trust proxy', 1);


// SECURITY LAYER 1: Basic Security Headers and Protection
app.use(helmetConfig);
app.use(hpp);
app.disable('x-powered-by'); // Hide Express

// SECURITY LAYER 2: Request Size and DDoS Protection
app.use(requestSizeLimiter);
app.use(generalLimiter);

// SECURITY LAYER 3: CORS Configuration
app.use(cors(corsOptions));

// SECURITY LAYER 4: Input Sanitization
app.use(express.json({
  limit: '5mb',
  verify: (req, res, buf) => {
    // Additional JSON validation
    try {
      JSON.parse(buf);
    } catch (e) {
      securityLogger.warn('Invalid JSON payload', {
        ip: req.ip,
        url: req.originalUrl,
        error: e.message,
        timestamp: new Date()
      });
      throw new Error('Invalid JSON');
    }
  }
}));
app.use(express.urlencoded({ extended: true, limit: '5mb' }));
app.use(mongoSanitizer);
app.use(xssProtection);

// SECURITY LAYER 5: Security Audit and Monitoring
// Disable security audit in development mode
if (process.env.NODE_ENV === 'production') {
  app.use(securityAudit);
}

// SECURITY LAYER 6: Socket.IO with Authentication
const io = new Server(server, {
  cors: {
    origin: ['http://localhost:3000', 'http://localhost:3001', 'http://localhost:3002', 'https://kaaryasthan.edumetrix.uk', 'https://student.edumetrix.uk', 'https://teacher.edumetrix.uk'],
    methods: ["GET", "POST"],
    credentials: true
  },
  allowEIO3: false, // Force latest version
  transports: ['websocket', 'polling']
});

// Socket.IO Authentication Middleware
io.use(async (socket, next) => {
  try {
    const token = socket.handshake.auth.token;

    console.log('🔐 Socket authentication attempt');
    console.log('   Token received:', token ? 'Yes (length: ' + token.length + ')' : 'No');
    console.log('   Token value:', token);
    console.log('   IP:', socket.handshake.address);

    if (!token) {
      console.error('❌ Socket connection without token');
      securityLogger.warn('Socket connection without token', {
        ip: socket.handshake.address,
        timestamp: new Date()
      });
      return next(new Error('Authentication error'));
    }

    

    // Verify JWT token
    console.log('   Verifying JWT token...');
    const decoded = jwt.verify(token, process.env.JWT_SECRET);

    console.log('✅ Token verified successfully');
    console.log('   User ID:', decoded.userId);
    console.log('   User Type:', decoded.userType);

    socket.userId = decoded.userId;
    socket.userType = decoded.userType;

    securityLogger.info('Socket authenticated', {
      userId: decoded.userId,
      userType: decoded.userType,
      ip: socket.handshake.address,
      timestamp: new Date()
    });

    next();
  } catch (err) {
    console.error('❌ Socket authentication failed:', err.message);
    console.error('   Error name:', err.name);
    console.error('   Token (first 20 chars):', socket.handshake.auth.token?.substring(0, 20));

    securityLogger.error('Socket authentication failed', {
      error: err.message,
      errorName: err.name,
      ip: socket.handshake.address,
      timestamp: new Date()
    });
    next(new Error('Authentication error'));
  }
});

app.set('io', io);
let users = {}; // Stores { userId: socketId }

// SECURITY LAYER 7: Database Connection with Security
mongoose.connect(process.env.MONGODB_URI || 'mongodb://localhost:27017/edumetrix', {
  useNewUrlParser: true,
  useUnifiedTopology: true,
  maxPoolSize: 10, // Maintain up to 10 socket connections
  serverSelectionTimeoutMS: 5000, // Keep trying to send operations for 5 seconds
  socketTimeoutMS: 45000, // Close sockets after 45 seconds of inactivity
  family: 4 // Use IPv4, skip trying IPv6
});

const db = mongoose.connection;
db.on('error', (error) => {
  securityLogger.error('MongoDB connection error', { error: error.message, timestamp: new Date() });
});
db.once('open', () => {
  securityLogger.info('Connected to MongoDB', { timestamp: new Date() });
});

// SECURITY LAYER 8: Static File Serving with Protection
app.use('/uploads', express.static(path.join(process.cwd(), 'uploads'), {
  dotfiles: 'deny',
  etag: false,
  extensions: ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'zip', 'rar', 'txt', 'mp4', 'mp3', 'wav', 'avi', 'mov', 'webm'],
  index: false,
  maxAge: '1d',
  redirect: false,
  setHeaders: (res, filePath, stat) => {
    res.setHeader('X-Content-Type-Options', 'nosniff');
    // Don't set X-Frame-Options for media files to allow inline display
    const ext = path.extname(filePath).toLowerCase();
    if (!['.jpg', '.jpeg', '.png', '.gif', '.webp', '.mp4', '.mp3', '.wav', '.webm'].includes(ext)) {
      res.setHeader('X-Frame-Options', 'DENY');
    }
    res.setHeader('Referrer-Policy', 'no-referrer');
    // More permissive CSP for media files
    if (['.jpg', '.jpeg', '.png', '.gif', '.webp'].includes(ext)) {
      res.setHeader('Content-Security-Policy', "default-src 'none';");
    } else if (['.mp4', '.mp3', '.wav', '.webm', '.avi', '.mov'].includes(ext)) {
      res.setHeader('Content-Security-Policy', "default-src 'none';");
    }
  }
}));

// SECURITY LAYER 9: Route-Specific Security
// Authentication routes with brute force protection
app.use('/api/auth/login', authLimiter, authBruteForce, bruteForceProtection);
app.use('/api/auth/admin/request-otp', otpLimiter, validateInput([validationSchemas.email]));
app.use('/api/auth/admin/verify-otp', authLimiter, validateInput([validationSchemas.email, validationSchemas.otp]));

// Admin routes with strict protection
app.use('/api/admin', auth, requireAdmin, strictLimiter, adminRouteProtection);

// Teacher routes with authentication
app.use('/api/teacher', (req, res, next) => {

  // Normal auth for other routes
  auth(req, res, (err) => {
    if (err) return next(err);
    if (req.user.userType !== 'teacher') {
      return res.status(403).json({ error: 'Teacher access required' });
    }
    next();
  });
});

// Student routes with authentication
app.use('/api/student', (req, res, next) => {
  console.log('Student route middleware - path:', req.path, 'method:', req.method);


  // Normal auth for other routes
  auth(req, res, (err) => {
    if (err) return next(err);
    if (req.user.userType !== 'student') {
      return res.status(403).json({ error: 'Student access required' });
    }
    next();
  });
});

// SECURITY LAYER 10: Route Definitions
app.use("/api/auth", authRoutes);
app.use("/api/admin", adminRoutes);
app.use("/api/admin/exams", adminExamsRoutes);
app.use("/api/admin/settings", adminSettingsRoutes);

app.use('/api/common', commonRoutes);
app.use("/api/teacher", teacherRoutes);
app.use("/api/teacher/exams", teacherExamsRoutes);
app.use("/api/teacher/assignments", teacherAssignmentsRoutes);
app.use("/api/teacher/quicktest", teacherQuicktestRoutes);
app.use("/api/teacher/homework", teacherHomeworkRoutes);
app.use("/api/teacher/settings", teacherSettingsRoutes);
app.use("/api/teacher/classes", classesRoutes);
app.use("/api/teacher/attendance", teacherAttendanceRoutes);
app.use("/api/teacher/reports", teacherReportsRoutes);

app.use("/api/student", studentRoutes);
app.use("/api/student/exams", studentExamsRoutes);
app.use("/api/student/assignments", studentAssignmentsRoutes);
app.use("/api/student/quicktest", studentQuicktestRoutes);
app.use("/api/student/homework", studentHomeworkRoutes);
app.use("/api/student/settings", studentSettingsRoutes);
app.use("/api/student/attendance", studentAttendanceRoutes);

// ✅ Chat & Misc routes
app.use("/api/chat", chatRoutes);
app.use("/api/classes", classesRoutes);
app.use("/api/finance", financeRoutes);
app.use("/api/reports", reportsRoutes);

// ✅ Message routes
app.use("/api/message", messageRouter(io, users));
app.use("/api/admin/message", adminMessageRoutes(io, users));

// ✅ Notification routes
app.use("/api/notifications", notificationRoutes);
app.use("/api/test-notification", testNotificationRoutes);

// SECURITY LAYER 11: Secure Socket.IO Events
io.on('connection', (socket) => {
  const userId = socket.userId; // set by auth middleware
  const userType = socket.userType;
  securityLogger.info('User connected', {
    socketId: socket.id,
    userId: socket.userId,
    userType: socket.userType,
    ip: socket.handshake.address,
    timestamp: new Date()
  });

  // Join user-specific room for notifications
  const userRoom = `user-${userId}`;
  socket.join(userRoom);
  console.log(`✅ User ${userId} (${userType}) joined room: ${userRoom}`);
  console.log(`   Socket ID: ${socket.id}`);
  console.log(`   Timestamp: ${new Date().toISOString()}`);

  // Join role-based rooms
  if (userType === 'admin') {
    socket.join('admins');
    console.log(`   Also joined: admins`);
  } else if (userType === 'teacher') {
    socket.join('teachers');
    console.log(`   Also joined: teachers`);
  } else if (userType === 'student') {
    socket.join('students');
    console.log(`   Also joined: students`);
  }

  // Secure room joining with validation
  // socket.on('join-chat', ({ userId, userType }) => {
  //   if (socket.userId === userId && socket.userType === userType) {
  //     socket.join(`${userType}-${userId}`);
  //     securityLogger.info('User joined chat room', { userId, userType, timestamp: new Date() });
  //   } else {
  //     securityLogger.warn('Unauthorized room join attempt', {
  //       attemptedUserId: userId,
  //       actualUserId: socket.userId,
  //       ip: socket.handshake.address,
  //       timestamp: new Date()
  //     });
  //   }
  // });

  // socket.on('join-chat', ({ chatId }) => {
  //   if (!chatId) return;
  //   socket.join(`chat-${chatId}`);
  //   securityLogger.info('User joined chat room', { userId, userType, chatId, timestamp: new Date() });
  // });

  socket.on('join-exam', ({ examId, userId, userType }) => {
    if (socket.userId === userId && socket.userType === userType) {
      socket.join(`exam-${examId}`);
      securityLogger.info('User joined exam', { examId, userId, userType, timestamp: new Date() });
    }
  });

  socket.on('leave-exam', ({ examId, userId }) => {
    if (socket.userId === userId) {
      socket.leave(`exam-${examId}`);
      securityLogger.info('User left exam', { examId, userId, timestamp: new Date() });
    }
  });

  socket.on('join-assignment', ({ assignmentId, userId, userType }) => {
    if (socket.userId === userId && socket.userType === userType) {
      socket.join(`assignment-${assignmentId}`);
      securityLogger.info('User joined assignment', { assignmentId, userId, userType, timestamp: new Date() });
    }
  });

  socket.on('leave-assignment', ({ assignmentId, userId }) => {
    if (socket.userId === userId) {
      socket.leave(`assignment-${assignmentId}`);
      securityLogger.info('User left assignment', { assignmentId, userId, timestamp: new Date() });
    }
  });

  socket.on('join-quicktest', ({ testId, userId, userType }) => {
    if (socket.userId === userId && socket.userType === userType) {
      socket.join(`quicktest-${testId}`);
      securityLogger.info('User joined quicktest', { testId, userId, userType, timestamp: new Date() });
    }
  });

  socket.on('leave-quicktest', ({ testId, userId }) => {
    if (socket.userId === userId) {
      socket.leave(`quicktest-${testId}`);
      securityLogger.info('User left quicktest', { testId, userId, timestamp: new Date() });
    }
  });

  socket.on('join-class', ({ classId, userId, userType }) => {
    if (socket.userId === userId && socket.userType === userType) {
      socket.join(`class-${classId}`);
      securityLogger.info('User joined class', { classId, userId, userType, timestamp: new Date() });
    }
  });

  // Chat room management
  socket.on('join-chat', ({ chatId }) => {
    if (!chatId) return;
    socket.join(`chat-${chatId}`);
    securityLogger.info('User joined chat room', { userId, userType, chatId, timestamp: new Date() });
  });

  socket.on('leave-chat', ({ chatId }) => {
    if (!chatId) return;
    socket.leave(`chat-${chatId}`);
    securityLogger.info('User left chat room', { userId, userType, chatId, timestamp: new Date() });
  });

  socket.on('typing', ({ chatId, typing }) => {
    if (!chatId) return;
    socket.to(`chat-${chatId}`).emit('typing', { chatId, userId, typing });
  });
  // socket.on('send-message', ({ from, to, message, userType }) => {
  //   if (socket.userId === from) {
  //     // Additional message validation
  //     if (typeof message === 'string' && message.length < 1000) {
  //       io.to(`${userType}-${to}`).emit('receive-message', {
  //         from,
  //         message: message.substring(0, 500), // Limit message length
  //         timestamp: new Date()
  //       });
  //       securityLogger.info('Message sent', { from, to, timestamp: new Date() });
  //     }
  //   }
  // });

  // socket.on('disconnect', () => {
  //   securityLogger.info('User disconnected', { 
  //     socketId: socket.id, 
  //     userId: socket.userId,
  //     timestamp: new Date() 
  //   });
  // });

  // 🔹 Send message
  // socket.on('send-message', async ({ chatId, message }) => {
  //   try {
  //     if (typeof message !== 'string' || !message.trim()) return;

  //     // 1. Save message to DB
  //     const chat = await Chat.findById(chatId);
  //     if (!chat) {
  //       return socket.emit('error', { message: 'Chat not found' });
  //     }

  //     const newMsg = {
  //       senderId: userId,
  //       senderType: userType,
  //       message: message.substring(0, 500),
  //       timestamp: new Date()
  //     };

  //     chat.messages.push(newMsg);
  //     chat.lastActivity = new Date();
  //     await chat.save();

  //     // 2. Emit to chat room
  //     io.to(`chat-${chatId}`).emit('newMessage', {
  //       chatId,
  //       message: {
  //         ...newMsg,
  //         senderName: socket.userType === 'admin'
  //           ? 'Admin'
  //           : socket.userType === 'teacher'
  //             ? 'Teacher'
  //             : 'Student'
  //       }
  //     });

  //     // 3. Handle role-based broadcast rules
  //     if (userType === 'admin') {
  //       // Admin → broadcast to all students & teachers
  //       io.to('students').emit('messageNotification', { from: 'Admin', chatId, preview: message });
  //       io.to('teachers').emit('messageNotification', { from: 'Admin', chatId, preview: message });
  //     } else if (userType === 'student') {
  //       // Student → notify admins & their teachers
  //       io.to('admins').emit('messageNotification', { from: 'Student', chatId, preview: message });
  //       // Optional: notify specific teachers (requires teacherId list from DB)
  //     } else if (userType === 'teacher') {
  //       // Teacher → notify admins & their students
  //       io.to('admins').emit('messageNotification', { from: 'Teacher', chatId, preview: message });
  //       // Optional: notify specific students (requires studentId list from DB)
  //     }

  //     securityLogger.info('Message sent', { from: userId, chatId, timestamp: new Date() });
  //   } catch (err) {
  //     console.error('Socket send-message error:', err);
  //     socket.emit('error', { message: 'Failed to send message' });
  //   }
  // });

  //new code---->>
  // socket.on('send-message', async ({ chatId, message }) => {
  //   try {
  //     if (typeof message !== 'string' || !message.trim()) return;

  //     const { newMsg } = await sendMessage({
  //       chatId,
  //       senderId: socket.userId,
  //       senderType: socket.userType,
  //       message,
  //       io
  //     });

  //     // ✅ Optional: role-based notifications (keep only if you want extra alerts)
  //     if (socket.userType === 'admin') {
  //       io.to('students').emit('messageNotification', { from: 'Admin', chatId, preview: newMsg.message });
  //       io.to('teachers').emit('messageNotification', { from: 'Admin', chatId, preview: newMsg.message });
  //     } else if (socket.userType === 'student') {
  //       io.to('admins').emit('messageNotification', { from: 'Student', chatId, preview: newMsg.message });
  //     } else if (socket.userType === 'teacher') {
  //       io.to('admins').emit('messageNotification', { from: 'Teacher', chatId, preview: newMsg.message });
  //     }

  //     securityLogger.info('Message sent via socket', { from: socket.userId, chatId });
  //   } catch (err) {
  //     console.error('Socket send-message error:', err);
  //     socket.emit('error', { message: err.message || 'Failed to send message' });
  //   }
  // });

  //   // 🔹 Disconnect
  //   socket.on('disconnect', () => {
  //     securityLogger.info('User disconnected', {
  //       socketId: socket.id,
  //       userId,
  //       timestamp: new Date()
  //     });
  //   });
  socket.on('register_user', (userId) => {
    console.log('📝 Received register_user event for userId:', userId);
    console.log('   Socket ID:', socket.id);
    console.log('   Socket user type:', socket.userType);

    if (!userId) {
      console.error('❌ Cannot register user: userId is null or undefined');
      return;
    }

    const userIdStr = userId.toString();

    if (!users[userIdStr]) {
      users[userIdStr] = new Set(); // Store socket IDs in a Set
      console.log(`✅ Created new user entry for ${userIdStr}`);
    }

    users[userIdStr].add(socket.id);
    console.log(`✅ User ${userIdStr} registered on socket ${socket.id}`);
    console.log(`   Total sockets for this user: ${users[userIdStr].size}`);
    console.log('   All registered users:', Object.keys(users));
  });

  const getConversationId = (id1, id2) => {
    return [id1, id2].sort().join('-'); // ✅ Always sorts alphabetically
  };

  socket.on("join_conversation", ({ senderId, receiverId }) => {
    const conversationId = getConversationId(senderId, receiverId);
    socket.join(conversationId);
    console.log(`✅ User joined conversation: ${conversationId}`);
  });

  // Handle direct messages (Real-time only, no database operation)
  socket.on('send_message', (messageData) => {
    const { conversationId, receiverId } = messageData;
    // Emit message to all users in the conversation room
    console.log(`Message received from sender to ${receiverId} in conversation ${conversationId}`);

    io.to(conversationId).emit('receive_message', messageData);

    // Also send the message to all active sockets of the receiver
    if (users[receiverId]) {
      users[receiverId].forEach((socketId) => {
        io.to(socketId).emit("receive_message", messageData);
      });
      console.log(`📨 Message delivered to receiver ${receiverId}`);
    } else {
      console.log(`❌ Receiver ${receiverId} is offline`);
    }
  });

  // Handle broadcast messages - DISABLED: Backend API routes handle emission
  // This was causing double emission (API route + socket handler)
  // socket.on('send_broadcast', (messageData) => {
  //   const { senderType } = messageData;
  //
  //   if (senderType === 'Admin') {
  //     console.log('Broadcast message from Admin>>', messageData);
  //     io.emit('receive_broadcast', messageData); // Send to all connected clients
  //   } else {
  //     console.log('Unauthorized broadcast attempt by:', senderType);
  //   }
  // });

  // Typing indicator (optional)
  socket.on('typing', ({ conversationId, senderId }) => {
    socket.to(conversationId).emit('user_typing', { senderId });
  });

  // Handle disconnection
  socket.on('disconnect', () => {
    console.log('User disconnected:', socket.id);

    // Remove socket ID from users object
    for (let userId in users) {
      if (users[userId].has(socket.id)) {
        users[userId].delete(socket.id);
        if (users[userId].size === 0) {
          delete users[userId];
        }
        break;
      }
    }
  });
});

// SECURITY LAYER 12: Error Handling
app.use((err, req, res, next) => {
  securityLogger.error('Application error', {
    error: err.message,
    stack: err.stack,
    ip: req.ip,
    url: req.originalUrl,
    method: req.method,
    timestamp: new Date()
  });

  // Don't leak error details in production
  const isDev = process.env.NODE_ENV === 'development';
  res.status(err.status || 500).json({
    error: isDev ? err.message : 'Internal server error',
    ...(isDev && { stack: err.stack })
  });
});

// SECURITY LAYER 13: 404 Handler
app.use('*', (req, res) => {
  securityLogger.warn('404 - Route not found', {
    ip: req.ip,
    url: req.originalUrl,
    method: req.method,
    userAgent: req.get('User-Agent'),
    timestamp: new Date()
  });
  res.status(404).json({ error: 'Route not found' });
});

// SECURITY LAYER 14: Graceful Shutdown
process.on('SIGTERM', () => {
  securityLogger.info('SIGTERM received. Shutting down gracefully', { timestamp: new Date() });
  server.close(() => {
    mongoose.connection.close();
    process.exit(0);
  });
});

process.on('SIGINT', () => {
  securityLogger.info('SIGINT received. Shutting down gracefully', { timestamp: new Date() });
  server.close(() => {
    mongoose.connection.close();
    process.exit(0);
  });
});

const PORT = process.env.PORT || 5001;
server.listen(PORT, () => {
  securityLogger.info('Server started', { port: PORT, timestamp: new Date() });
});