import nodemailer from 'nodemailer';
import dotenv from "dotenv";

// In-memory OTP storage (in production, use Redis or database)
const otpStore = new Map();
dotenv.config();

// Create transporter for Gmail
console.log('EMAIL_USER:', process.env.EMAIL_USER);
console.log('EMAIL_PASSWORD:', process.env.EMAIL_PASSWORD);
export const createTransporter = () => {
  return nodemailer.createTransport({
    host: 'smtp.gmail.com',
    port: 465,
    secure: true,
    auth: {
      user: process.env.EMAIL_USER,
      pass: process.env.EMAIL_PASSWORD,
    },
  });
};

// Generate 6-digit OTP
const generateOTP = () => {
  return Math.floor(100000 + Math.random() * 900000).toString();
};

// Send OTP via email
const sendOTP = async (email) => {
  try {
    // Only allow edumetrixuk@gmail.com for admin login
    const allowedAdmins = ['edumetrixuk@gmail.com', 'dilkashpeshimam@gmail.com'];

    if (!allowedAdmins.includes(email)) {
      return res.status(403).json({ message: 'Unauthorized email address for admin access' });
    }

    // For development, use a fixed OTP
    // const otp = process.env.NODE_ENV === 'production' ? generateOTP() : '123456';
    const otp = generateOTP();


    // Store OTP with expiration (5 minutes)
    otpStore.set(email, {
      otp,
      expiresAt: Date.now() + 5 * 60 * 1000, // 5 minutes
      attempts: 0,
    });


    // // For demo/development purposes - just log the OTP
    // // In production, configure EMAIL_USER and EMAIL_PASSWORD environment variables
    // console.log(`\n🔑 Development OTP for ${email}: ${otp}`);
    // console.log(`⏰ Expires at: ${new Date(Date.now() + 5 * 60 * 1000).toLocaleTimeString()}\n`);
    // return { success: true, message: `OTP sent successfully. For development, use: ${otp}` };

    // Production email sending
    const transporter = createTransporter();
    const mailOptions = {
      from: process.env.EMAIL_USER || 'edumetrixuk@gmail.com',
      to: email,
      subject: 'EduMetrix Admin Login - OTP Verification',
      html: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="text-align: center; margin-bottom: 30px;">
            <h1 style="color: #1976d2; margin: 0;">EduMetrix</h1>
            <p style="color: #666; margin: 5px 0;">Educational Management Platform</p>
          </div>
          
          <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; border-radius: 12px; text-align: center; margin-bottom: 30px;">
            <h2 style="margin: 0 0 10px 0;">Admin Login Verification</h2>
            <p style="margin: 0; opacity: 0.9;">Your one-time password for admin access</p>
          </div>

          <div style="background: #f8f9fa; padding: 30px; border-radius: 12px; text-align: center; margin-bottom: 30px;">
            <p style="margin: 0 0 20px 0; color: #666;">Enter this code to complete your login:</p>
            <div style="background: white; border: 2px solid #1976d2; padding: 20px; border-radius: 8px; display: inline-block;">
              <span style="font-size: 32px; font-weight: bold; color: #1976d2; letter-spacing: 8px; font-family: 'Courier New', monospace;">${otp}</span>
            </div>
            <p style="margin: 20px 0 0 0; color: #999; font-size: 14px;">This code expires in 5 minutes</p>
          </div>

          <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 20px; border-radius: 8px; margin-bottom: 30px;">
            <p style="margin: 0; color: #856404; font-size: 14px;">
              <strong>Security Notice:</strong> This OTP is only valid for the authorized admin email address. 
              If you did not request this login, please ignore this email.
            </p>
          </div>

          <div style="text-align: center; color: #999; font-size: 12px; border-top: 1px solid #eee; padding-top: 20px;">
            <p style="margin: 0;">This is an automated message from EduMetrix Platform</p>
            <p style="margin: 5px 0 0 0;">Please do not reply to this email</p>
          </div>
        </div>
      `,
    };

    await transporter.sendMail(mailOptions);
    return { success: true, message: 'OTP sent successfully' };
  } catch (error) {
    console.error('Email send error:', error);
    throw new Error(error.message || 'Failed to send OTP email');
  }
};

// Verify OTP
const verifyOTP = (email, inputOTP) => {
  const storedData = otpStore.get(email);

  if (!storedData) {
    return { success: false, message: 'OTP not found or expired' };
  }

  // Check if expired
  if (Date.now() > storedData.expiresAt) {
    otpStore.delete(email);
    return { success: false, message: 'OTP has expired' };
  }

  // Check attempts limit
  if (storedData.attempts >= 3) {
    otpStore.delete(email);
    return { success: false, message: 'Too many failed attempts. Please request a new OTP' };
  }

  // Check OTP match
  if (storedData.otp !== inputOTP) {
    storedData.attempts += 1;
    return { success: false, message: 'Invalid OTP' };
  }

  // OTP verified successfully
  otpStore.delete(email);
  return { success: true, message: 'OTP verified successfully' };
};

// Clean expired OTPs (run periodically)
const cleanExpiredOTPs = () => {
  const now = Date.now();
  for (const [email, data] of otpStore.entries()) {
    if (now > data.expiresAt) {
      otpStore.delete(email);
    }
  }
};

// Clean expired OTPs every 5 minutes
setInterval(cleanExpiredOTPs, 5 * 60 * 1000);

export {
  sendOTP,
  verifyOTP
};
