const request = require('supertest');
const express = require('express');
const mongoose = require('mongoose');
const bcrypt = require('bcryptjs');
const User = require('../../models/admin/User');
const authRoutes = require('../../routes/admin/auth');

const app = express();
app.use(express.json());
app.use('/api/auth', authRoutes);

describe('Authentication Integration Tests', () => {
  beforeAll(async () => {
    await mongoose.connect('mongodb://localhost:27017/edumetrix_test', {
      useNewUrlParser: true,
      useUnifiedTopology: true,
    });
  });

  afterAll(async () => {
    await mongoose.connection.close();
  });

  beforeEach(async () => {
    await User.deleteMany({});
    
    // Create test admin user
    const hashedPassword = await bcrypt.hash('admin123', 10);
    await User.create({
      username: 'admin@edumetrix.uk',
      password: hashedPassword,
      userType: 'admin',
      profile: {
        name: 'EduMetrix',
        email: 'admin@edumetrix.uk'
      }
    });
  });

  afterEach(async () => {
    await User.deleteMany({});
  });

  describe('POST /api/auth/admin/request-otp', () => {
    test('should send OTP for valid admin email', async () => {
      const response = await request(app)
        .post('/api/auth/admin/request-otp')
        .send({
          email: 'edumetrixuk@gmail.com'
        });

      expect(response.status).toBe(200);
      expect(response.body.success).toBe(true);
      expect(response.body.message).toContain('OTP');
    });

    test('should reject invalid email', async () => {
      const response = await request(app)
        .post('/api/auth/admin/request-otp')
        .send({
          email: 'invalid@example.com'
        });

      expect(response.status).toBe(403);
      expect(response.body.message).toContain('Unauthorized email address');
    });

    test('should validate email format', async () => {
      const response = await request(app)
        .post('/api/auth/admin/request-otp')
        .send({
          email: 'invalid-email'
        });

      expect(response.status).toBe(400);
    });
  });

  describe('POST /api/auth/login', () => {
    beforeEach(async () => {
      // Create test student user
      const hashedPassword = await bcrypt.hash('student123', 10);
      await User.create({
        username: 'student@test.com',
        password: hashedPassword,
        userType: 'student',
        profile: {
          name: 'Test Student',
          email: 'student@test.com'
        }
      });
    });

    test('should login valid student user', async () => {
      const response = await request(app)
        .post('/api/auth/login')
        .send({
          username: 'student@test.com',
          password: 'student123'
        });

      expect(response.status).toBe(200);
      expect(response.body.token).toBeDefined();
      expect(response.body.user.userType).toBe('student');
    });

    test('should reject admin login via regular endpoint', async () => {
      const response = await request(app)
        .post('/api/auth/login')
        .send({
          username: 'admin@edumetrix.uk',
          password: 'admin123'
        });

      expect(response.status).toBe(403);
      expect(response.body.message).toContain('Admin login not allowed');
    });

    test('should reject invalid credentials', async () => {
      const response = await request(app)
        .post('/api/auth/login')
        .send({
          username: 'student@test.com',
          password: 'wrongpassword'
        });

      expect(response.status).toBe(400);
      expect(response.body.message).toBe('Invalid credentials');
    });

    test('should reject missing credentials', async () => {
      const response = await request(app)
        .post('/api/auth/login')
        .send({
          username: 'student@test.com'
          // Missing password
        });

      expect(response.status).toBe(400);
      expect(response.body.message).toContain('Please provide username and password');
    });
  });
});