const crypto = require('crypto');
const argon2 = require('argon2');

class EnterpriseEncryption {
  constructor() {
    this.algorithm = 'aes-256-gcm';
    this.keyLength = 32; // 256 bits
    this.ivLength = 16; // 128 bits
    this.tagLength = 16; // 128 bits
    this.saltLength = 32; // 256 bits
    
    // Initialize master key from environment or generate one
    this.masterKey = this.initializeMasterKey();
  }

  initializeMasterKey() {
    const envKey = process.env.MASTER_ENCRYPTION_KEY;
    
    if (envKey) {
      // Validate the key length
      const keyBuffer = Buffer.from(envKey, 'base64');
      if (keyBuffer.length !== this.keyLength) {
        throw new Error('Invalid master encryption key length');
      }
      return keyBuffer;
    }
    
    // Generate a new key for development (should use proper key management in production)
    if (process.env.NODE_ENV !== 'production') {
      console.warn('⚠️  Using auto-generated encryption key. Set MASTER_ENCRYPTION_KEY in production!');
      return crypto.randomBytes(this.keyLength);
    }
    
    throw new Error('MASTER_ENCRYPTION_KEY environment variable required in production');
  }

  // Derive key from master key and salt using PBKDF2
  deriveKey(salt, iterations = 100000) {
    return crypto.pbkdf2Sync(this.masterKey, salt, iterations, this.keyLength, 'sha256');
  }

  // Encrypt sensitive data
  encrypt(plaintext, context = '') {
    try {
      if (!plaintext) {
        throw new Error('Plaintext is required for encryption');
      }

      // Generate random salt and IV
      const salt = crypto.randomBytes(this.saltLength);
      const iv = crypto.randomBytes(this.ivLength);
      
      // Derive encryption key
      const key = this.deriveKey(salt);
      
      // Create cipher
      const cipher = crypto.createCipher(this.algorithm, key);
      cipher.setAAD(Buffer.from(context, 'utf8')); // Additional Authenticated Data
      
      // Encrypt
      let encrypted = cipher.update(plaintext, 'utf8');
      encrypted = Buffer.concat([encrypted, cipher.final()]);
      
      // Get authentication tag
      const tag = cipher.getAuthTag();
      
      // Combine all components
      const result = Buffer.concat([salt, iv, tag, encrypted]);
      
      return {
        encrypted: result.toString('base64'),
        algorithm: this.algorithm,
        keyDerivation: 'pbkdf2',
        iterations: 100000
      };
    } catch (error) {
      throw new Error(`Encryption failed: ${error.message}`);
    }
  }

  // Decrypt sensitive data
  decrypt(encryptedData, context = '') {
    try {
      if (!encryptedData || !encryptedData.encrypted) {
        throw new Error('Encrypted data is required for decryption');
      }

      const data = Buffer.from(encryptedData.encrypted, 'base64');
      
      // Extract components
      const salt = data.slice(0, this.saltLength);
      const iv = data.slice(this.saltLength, this.saltLength + this.ivLength);
      const tag = data.slice(this.saltLength + this.ivLength, this.saltLength + this.ivLength + this.tagLength);
      const encrypted = data.slice(this.saltLength + this.ivLength + this.tagLength);
      
      // Derive decryption key
      const key = this.deriveKey(salt, encryptedData.iterations || 100000);
      
      // Create decipher
      const decipher = crypto.createDecipher(encryptedData.algorithm || this.algorithm, key);
      decipher.setAAD(Buffer.from(context, 'utf8'));
      decipher.setAuthTag(tag);
      
      // Decrypt
      let decrypted = decipher.update(encrypted);
      decrypted = Buffer.concat([decrypted, decipher.final()]);
      
      return decrypted.toString('utf8');
    } catch (error) {
      throw new Error(`Decryption failed: ${error.message}`);
    }
  }

  // Hash passwords using Argon2id (latest standard)
  async hashPassword(password, options = {}) {
    const config = {
      type: argon2.argon2id,
      memoryCost: options.memoryCost || 65536, // 64 MB
      timeCost: options.timeCost || 3, // 3 iterations
      parallelism: options.parallelism || 4, // 4 threads
      hashLength: options.hashLength || 32, // 32 bytes
      saltLength: options.saltLength || 32, // 32 bytes
      ...options
    };

    try {
      const hash = await argon2.hash(password, config);
      return {
        hash,
        algorithm: 'argon2id',
        config: {
          memoryCost: config.memoryCost,
          timeCost: config.timeCost,
          parallelism: config.parallelism
        }
      };
    } catch (error) {
      throw new Error(`Password hashing failed: ${error.message}`);
    }
  }

  // Verify password against hash
  async verifyPassword(password, hashedPassword) {
    try {
      const hashToVerify = typeof hashedPassword === 'object' 
        ? hashedPassword.hash 
        : hashedPassword;
      
      return await argon2.verify(hashToVerify, password);
    } catch (error) {
      throw new Error(`Password verification failed: ${error.message}`);
    }
  }

  // Generate cryptographically secure random tokens
  generateSecureToken(length = 32) {
    return crypto.randomBytes(length).toString('base64url');
  }

  // Generate secure random passwords
  generateSecurePassword(length = 16, options = {}) {
    const {
      includeUppercase = true,
      includeLowercase = true,
      includeNumbers = true,
      includeSymbols = true,
      excludeSimilar = true // Exclude similar looking characters
    } = options;

    let charset = '';
    
    if (includeUppercase) charset += 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    if (includeLowercase) charset += 'abcdefghijklmnopqrstuvwxyz';
    if (includeNumbers) charset += '0123456789';
    if (includeSymbols) charset += '!@#$%^&*()_+-=[]{}|;:,.<>?';
    
    if (excludeSimilar) {
      // Remove similar looking characters
      charset = charset.replace(/[0O1lI]/g, '');
    }

    if (!charset) {
      throw new Error('At least one character type must be included');
    }

    let password = '';
    for (let i = 0; i < length; i++) {
      const randomIndex = crypto.randomInt(0, charset.length);
      password += charset[randomIndex];
    }

    return password;
  }

  // Key derivation for API keys and tokens
  deriveApiKey(userId, purpose, expiresAt = null) {
    const data = `${userId}:${purpose}:${expiresAt || 'never'}:${Date.now()}`;
    const hmac = crypto.createHmac('sha256', this.masterKey);
    hmac.update(data);
    
    return {
      keyId: crypto.randomUUID(),
      key: hmac.digest('base64url'),
      purpose,
      userId,
      expiresAt,
      createdAt: new Date().toISOString()
    };
  }

  // Field-level encryption for database
  encryptField(value, fieldName) {
    if (!value) return null;
    
    const context = `field:${fieldName}`;
    return this.encrypt(String(value), context);
  }

  decryptField(encryptedValue, fieldName) {
    if (!encryptedValue) return null;
    
    const context = `field:${fieldName}`;
    return this.decrypt(encryptedValue, context);
  }

  // PII (Personally Identifiable Information) encryption
  encryptPII(data) {
    const encrypted = {};
    const piiFields = [
      'email', 'phone', 'address', 'ssn', 'dateOfBirth',
      'guardianName', 'guardianPhone', 'guardianEmail'
    ];

    for (const [key, value] of Object.entries(data)) {
      if (piiFields.includes(key) && value) {
        encrypted[key] = this.encryptField(value, key);
      } else {
        encrypted[key] = value;
      }
    }

    return encrypted;
  }

  decryptPII(data) {
    const decrypted = {};
    const piiFields = [
      'email', 'phone', 'address', 'ssn', 'dateOfBirth',
      'guardianName', 'guardianPhone', 'guardianEmail'
    ];

    for (const [key, value] of Object.entries(data)) {
      if (piiFields.includes(key) && value && typeof value === 'object') {
        try {
          decrypted[key] = this.decryptField(value, key);
        } catch (error) {
          console.warn(`Failed to decrypt field ${key}:`, error.message);
          decrypted[key] = null;
        }
      } else {
        decrypted[key] = value;
      }
    }

    return decrypted;
  }

  // Create secure session tokens
  createSessionToken(userId, sessionData = {}) {
    const payload = {
      userId,
      sessionId: crypto.randomUUID(),
      createdAt: Date.now(),
      ...sessionData
    };

    const token = this.encrypt(JSON.stringify(payload), 'session');
    
    return {
      token: token.encrypted,
      sessionId: payload.sessionId,
      expiresAt: Date.now() + (24 * 60 * 60 * 1000) // 24 hours
    };
  }

  verifySessionToken(tokenData) {
    try {
      const decrypted = this.decrypt({ encrypted: tokenData }, 'session');
      const payload = JSON.parse(decrypted);
      
      // Check if token is expired (if it has expiration)
      if (payload.expiresAt && Date.now() > payload.expiresAt) {
        throw new Error('Session token expired');
      }
      
      return payload;
    } catch (error) {
      throw new Error(`Invalid session token: ${error.message}`);
    }
  }

  // Data masking for logs and responses
  maskSensitiveData(data, fields = []) {
    const sensitiveFields = [
      'password', 'token', 'secret', 'key', 'creditCard',
      'ssn', 'socialSecurity', 'passport', ...fields
    ];

    const mask = (obj) => {
      if (typeof obj !== 'object' || obj === null) {
        return obj;
      }

      const masked = Array.isArray(obj) ? [] : {};

      for (const [key, value] of Object.entries(obj)) {
        const isMatch = sensitiveFields.some(field => 
          key.toLowerCase().includes(field.toLowerCase())
        );

        if (isMatch && typeof value === 'string') {
          masked[key] = this.maskString(value);
        } else if (typeof value === 'object') {
          masked[key] = mask(value);
        } else {
          masked[key] = value;
        }
      }

      return masked;
    };

    return mask(data);
  }

  maskString(str) {
    if (!str || str.length <= 4) {
      return '***';
    }
    
    const start = str.substring(0, 2);
    const end = str.substring(str.length - 2);
    const middle = '*'.repeat(Math.max(str.length - 4, 3));
    
    return start + middle + end;
  }

  // Generate cryptographic signatures
  sign(data, key = null) {
    const signingKey = key || this.masterKey;
    const hmac = crypto.createHmac('sha256', signingKey);
    hmac.update(JSON.stringify(data));
    return hmac.digest('base64url');
  }

  verifySignature(data, signature, key = null) {
    const signingKey = key || this.masterKey;
    const expectedSignature = this.sign(data, signingKey);
    return crypto.timingSafeEqual(
      Buffer.from(signature, 'base64url'),
      Buffer.from(expectedSignature, 'base64url')
    );
  }

  // Key rotation utilities
  rotateKey() {
    const newKey = crypto.randomBytes(this.keyLength);
    const oldKey = this.masterKey;
    
    return {
      newKey: newKey.toString('base64'),
      oldKey: oldKey.toString('base64'),
      rotatedAt: new Date().toISOString()
    };
  }
}

// Create singleton instance
const encryption = new EnterpriseEncryption();

module.exports = encryption;